<?php

/**
 *
 * Class Vc_IconPicker
 * @since 4.4
 * See example usage in shortcode 'vc_icon'
 *
 *      `` example
 *        array(
 *            'type' => 'iconpicker',
 *            'heading' => __( 'Icon', 'js_composer' ),
 *            'param_name' => 'icon_fontawesome',
 *            'settings' => array(
 *                'emptyIcon' => false, // default true, display an "EMPTY" icon? - if false it will display first icon
 *     from set as default.
 *                'iconsPerPage' => 200, // default 100, how many icons per/page to display
 *            ),
 *            'dependency' => array(
 *                'element' => 'type',
 *                'value' => 'fontawesome',
 *            ),
 *        ),
 * vc_filter: vc_iconpicker-type-{your_icon_font_name} - filter to add new icon font type. see example for
 *     vc_iconpicker-type-fontawesome in bottom of this file Also // SEE HOOKS FOLDER FOR FONTS REGISTERING/ENQUEUE IN
 *     BASE @path "/include/autoload/hook-vc-iconpicker-param.php"
 */
class Vc_IconPicker {
	/**
	 * @since 4.4
	 * @var array - save current param data array from vc_map
	 */
	protected $settings;
	/**
	 * @since 4.4
	 * @var string - save a current field value
	 */
	protected $value;
	/**
	 * @since 4.4
	 * @var array - optional, can be used as self source from self array., you can pass it also with filter see
	 *     Vc_IconPicker::setDefaults
	 */
	protected $source = array();

	/**
	 * @since 4.4
	 *
	 * @param $settings - param field data array
	 * @param $value - param field value
	 */
	public function __construct( $settings, $value ) {
		$this->settings = $settings;
		$this->setDefaults();

		$this->value = $value; // param field value
	}

	/**
	 * Set default function will extend current settings with defaults
	 * It can be used in Vc_IconPicker::render, but also it is passed to input field and was hooked in composer-atts.js
	 * file See vc.atts.iconpicker in wp-content/plugins/js_composer/assets/js/params/composer-atts.js init method
	 *  - it initializes javascript logic, you can provide ANY default param to it with 'settings' key
	 * @since 4.4
	 */
	protected function setDefaults() {
		if ( ! isset( $this->settings['settings'], $this->settings['settings']['type'] ) ) {
			$this->settings['settings']['type'] = 'fontawesome'; // Default type for icons
		}

		// More about this you can read in http://codeb.it/fonticonpicker/
		if ( ! isset( $this->settings['settings'], $this->settings['settings']['hasSearch'] ) ) {
			// Whether or not to show the search bar.
			$this->settings['settings']['hasSearch'] = true;
		}
		if ( ! isset( $this->settings['settings'], $this->settings['settings']['emptyIcon'] ) ) {
			// Whether or not empty icon should be shown on the icon picker
			$this->settings['settings']['emptyIcon'] = true;
		}
		if ( ! isset( $this->settings['settings'], $this->settings['settings']['allCategoryText'] ) ) {
			// If categorized then use this option
			$this->settings['settings']['allCategoryText'] = __( 'From all categories', 'js_composer' );
		}
		if ( ! isset( $this->settings['settings'], $this->settings['settings']['unCategorizedText'] ) ) {
			// If categorized then use this option
			$this->settings['settings']['unCategorizedText'] = __( 'Uncategorized', 'js_composer' );
		}

		/**
		 * Source for icons, can be passed via "mapping" or with filter vc_iconpicker-type-{your_type} (default fontawesome)
		 * vc_filter: vc_iconpicker-type-{your_type} (default fontawesome)
		 */
		if ( isset( $this->settings['settings'], $this->settings['settings']['source'] ) ) {
			$this->source = $this->settings['settings']['source'];
			unset( $this->settings['settings']['source'] ); // We don't need this on frontend.(js)
		}
	}

	/**
	 * Render edit form field type 'iconpicker' with selected settings and provided value.
	 * It uses javascript file vc-icon-picker (vc_iconpicker_base_register_js, vc_iconpicker_editor_jscss),
	 * see wp-content/plugins/js_composer/include/autoload/hook-vc-iconpicker-param.php folder
	 * @since 4.4
	 * @return string - rendered param field for editor panel
	 */
	public function render() {

		$output = '<div class="vc-iconpicker-wrapper"><select class="vc-iconpicker">';

		// call filter vc_iconpicker-type-{your_type}, e.g. vc_iconpicker-type-fontawesome with passed source from shortcode(default empty array). to get icons
		$arr = apply_filters( 'vc_iconpicker-type-' . esc_attr( $this->settings['settings']['type'] ), $this->source );

		if ( ! empty( $arr ) ) {
			foreach ( $arr as $group => $icons ) {
				if ( ! is_array( $icons ) || ! is_array( current( $icons ) ) ) {
					$class_key = key( $icons );
					$output .= '<option value="' . esc_attr( $class_key ) . '" ' . ( strcmp( $class_key, $this->value ) === 0 ? 'selected="selected"' : '' ) . '>' . esc_html( current( $icons ) ) . '</option>' . "\n";
				} else {
					$output .= '<optgroup label="' . esc_attr( $group ) . '">' . "\n";
					foreach ( $icons as $key => $label ) {
						$class_key = key( $label );
						$output .= '<option value="' . esc_attr( $class_key ) . '" ' . ( strcmp( $class_key, $this->value ) === 0 ? 'selected="selected"' : '' ) . '>' . esc_html( current( $label ) ) . '</option>' . "\n";
					}
					$output .= '</optgroup>' . "\n";
				}
			}
		}
		$output .= '</select></div>';

		$output .= '<input name="' .
		           esc_attr( $this->settings['param_name'] ) .
		           '" class="wpb_vc_param_value  ' .
		           esc_attr( $this->settings['param_name'] ) . ' ' .
		           esc_attr( $this->settings['type'] ) . '_field" type="hidden" value="' . esc_attr( $this->value ) . '" ' .
		           ( ( isset( $this->settings['settings'] ) && ! empty( $this->settings['settings'] ) ) ? ' data-settings="' . esc_attr( json_encode( $this->settings['settings'] ) ) . '" ' : '' ) .
		           ' />';

		return $output;
	}
}

/**
 * Function for rendering param in edit form (add element)
 * Parse settings from vc_map and entered values.
 *
 * @param $settings
 * @param $value
 * @param $tag
 *
 * @since 4.4
 * @return string - rendered template for params in edit form
 *
 */
function vc_iconpicker_form_field( $settings, $value, $tag ) {

	$icon_picker = new Vc_IconPicker( $settings, $value, $tag );

	return apply_filters( 'vc_iconpicker_render_filter', $icon_picker->render() );
}

// SEE HOOKS FOLDER FOR FONTS REGISTERING/ENQUEUE IN BASE @path "/include/autoload/hook-vc-iconpicker-param.php"

add_filter( 'vc_iconpicker-type-fontawesome', 'vc_iconpicker_type_fontawesome' );

/**
 * Fontawesome icons from FontAwesome :)
 *
 * @param $icons - taken from filter - vc_map param field settings['source'] provided icons (default empty array).
 * If array categorized it will auto-enable category dropdown
 *
 * @since 4.4
 * @return array - of icons for iconpicker, can be categorized, or not.
 */
function vc_iconpicker_type_fontawesome( $icons ) {
	// Categorized icons ( you can also output simple array ( key=> value ), where key = icon class, value = icon readable name ).
	$fontawesome_icons = array(
		"Web Application Icons" => array(
			array( "fa fa-adjust" => __( "Adjust", "js_composer" ) ),
			array( "fa fa-anchor" => __( "Anchor", "js_composer" ) ),
			array( "fa fa-archive" => __( "Archive", "js_composer" ) ),
			array( "fa fa-area-chart" => __( "Area Chart", "js_composer" ) ),
			array( "fa fa-arrows" => __( "Arrows", "js_composer" ) ),
			array( "fa fa-arrows-h" => __( "Arrows Horizontal", "js_composer" ) ),
			array( "fa fa-arrows-v" => __( "Arrows Vertical", "js_composer" ) ),
			array( "fa fa-asterisk" => __( "Asterisk", "js_composer" ) ),
			array( "fa fa-at" => __( "At", "js_composer" ) ),
			array( "fa fa-ban" => __( "Ban", "js_composer" ) ),
			array( "fa fa-bar-chart" => __( "Bar Chart", "js_composer" ) ),
			array( "fa fa-barcode" => __( "Barcode", "js_composer" ) ),
			array( "fa fa-bars" => __( "Bars", "js_composer" ) ),
			array( "fa fa-beer" => __( "Beer", "js_composer" ) ),
			array( "fa fa-bell" => __( "Bell", "js_composer" ) ),
			array( "fa fa-bell-o" => __( "Bell Outlined", "js_composer" ) ),
			array( "fa fa-bell-slash" => __( "Bell Slash", "js_composer" ) ),
			array( "fa fa-bell-slash-o" => __( "Bell Slash Outlined", "js_composer" ) ),
			array( "fa fa-bicycle" => __( "Bicycle", "js_composer" ) ),
			array( "fa fa-binoculars" => __( "Binoculars", "js_composer" ) ),
			array( "fa fa-birthday-cake" => __( "Birthday Cake", "js_composer" ) ),
			array( "fa fa-bolt" => __( "Lightning Bolt", "js_composer" ) ),
			array( "fa fa-bomb" => __( "Bomb", "js_composer" ) ),
			array( "fa fa-book" => __( "Book", "js_composer" ) ),
			array( "fa fa-bookmark" => __( "Bookmark", "js_composer" ) ),
			array( "fa fa-bookmark-o" => __( "Bookmark Outlined", "js_composer" ) ),
			array( "fa fa-briefcase" => __( "Briefcase", "js_composer" ) ),
			array( "fa fa-bug" => __( "Bug", "js_composer" ) ),
			array( "fa fa-building" => __( "Building", "js_composer" ) ),
			array( "fa fa-building-o" => __( "Building Outlined", "js_composer" ) ),
			array( "fa fa-bullhorn" => __( "bullhorn", "js_composer" ) ),
			array( "fa fa-bullseye" => __( "Bullseye", "js_composer" ) ),
			array( "fa fa-bus" => __( "Bus", "js_composer" ) ),
			array( "fa fa-calculator" => __( "Calculator", "js_composer" ) ),
			array( "fa fa-calendar" => __( "Calendar", "js_composer" ) ),
			array( "fa fa-calendar-o" => __( "Calendar-o", "js_composer" ) ),
			array( "fa fa-camera" => __( "Camera", "js_composer" ) ),
			array( "fa fa-camera-retro" => __( "Camera-retro", "js_composer" ) ),
			array( "fa fa-car" => __( "Car", "js_composer" ) ),
			array( "fa fa-caret-square-o-down" => __( "Caret Square Outlined Down", "js_composer" ) ),
			array( "fa fa-caret-square-o-left" => __( "Caret Square Outlined Left", "js_composer" ) ),
			array( "fa fa-caret-square-o-right" => __( "Caret Square Outlined Right", "js_composer" ) ),
			array( "fa fa-caret-square-o-up" => __( "Caret Square Outlined Up", "js_composer" ) ),
			array( "fa fa-cc" => __( "Closed Captions", "js_composer" ) ),
			array( "fa fa-certificate" => __( "Certificate", "js_composer" ) ),
			array( "fa fa-check" => __( "Check", "js_composer" ) ),
			array( "fa fa-check-circle" => __( "Check Circle", "js_composer" ) ),
			array( "fa fa-check-circle-o" => __( "Check Circle Outlined", "js_composer" ) ),
			array( "fa fa-check-square" => __( "Check Square", "js_composer" ) ),
			array( "fa fa-check-square-o" => __( "Check Square Outlined", "js_composer" ) ),
			array( "fa fa-child" => __( "Child", "js_composer" ) ),
			array( "fa fa-circle" => __( "Circle", "js_composer" ) ),
			array( "fa fa-circle-o" => __( "Circle Outlined", "js_composer" ) ),
			array( "fa fa-circle-o-notch" => __( "Circle Outlined Notched", "js_composer" ) ),
			array( "fa fa-circle-thin" => __( "Circle Outlined Thin", "js_composer" ) ),
			array( "fa fa-clock-o" => __( "Clock Outlined", "js_composer" ) ),
			array( "fa fa-cloud" => __( "Cloud", "js_composer" ) ),
			array( "fa fa-cloud-download" => __( "Cloud Download", "js_composer" ) ),
			array( "fa fa-cloud-upload" => __( "Cloud Upload", "js_composer" ) ),
			array( "fa fa-code" => __( "Code", "js_composer" ) ),
			array( "fa fa-code-fork" => __( "Code-fork", "js_composer" ) ),
			array( "fa fa-coffee" => __( "Coffee", "js_composer" ) ),
			array( "fa fa-cog" => __( "Cog", "js_composer" ) ),
			array( "fa fa-cogs" => __( "Cogs", "js_composer" ) ),
			array( "fa fa-comment" => __( "Comment", "js_composer" ) ),
			array( "fa fa-comment-o" => __( "Comment-o", "js_composer" ) ),
			array( "fa fa-comments" => __( "Comments", "js_composer" ) ),
			array( "fa fa-comments-o" => __( "Comments-o", "js_composer" ) ),
			array( "fa fa-compass" => __( "Compass", "js_composer" ) ),
			array( "fa fa-copyright" => __( "Copyright", "js_composer" ) ),
			array( "fa fa-credit-card" => __( "credit-card", "js_composer" ) ),
			array( "fa fa-crop" => __( "Crop", "js_composer" ) ),
			array( "fa fa-crosshairs" => __( "Crosshairs", "js_composer" ) ),
			array( "fa fa-cube" => __( "Cube", "js_composer" ) ),
			array( "fa fa-cubes" => __( "Cubes", "js_composer" ) ),
			array( "fa fa-cutlery" => __( "Cutlery", "js_composer" ) ),
			array( "fa fa-database" => __( "Database", "js_composer" ) ),
			array( "fa fa-desktop" => __( "Desktop", "js_composer" ) ),
			array( "fa fa-dot-circle-o" => __( "Dot Circle O", "js_composer" ) ),
			array( "fa fa-download" => __( "Download", "js_composer" ) ),
			array( "fa fa-ellipsis-h" => __( "Ellipsis Horizontal", "js_composer" ) ),
			array( "fa fa-ellipsis-v" => __( "Ellipsis Vertical", "js_composer" ) ),
			array( "fa fa-envelope" => __( "Envelope", "js_composer" ) ),
			array( "fa fa-envelope-o" => __( "Envelope Outlined", "js_composer" ) ),
			array( "fa fa-envelope-square" => __( "Envelope Square", "js_composer" ) ),
			array( "fa fa-eraser" => __( "Eraser", "js_composer" ) ),
			array( "fa fa-exchange" => __( "Exchange", "js_composer" ) ),
			array( "fa fa-exclamation" => __( "Exclamation", "js_composer" ) ),
			array( "fa fa-exclamation-circle" => __( "Exclamation Circle", "js_composer" ) ),
			array( "fa fa-exclamation-triangle" => __( "Exclamation Triangle", "js_composer" ) ),
			array( "fa fa-external-link" => __( "External Link", "js_composer" ) ),
			array( "fa fa-external-link-square" => __( "External Link Square", "js_composer" ) ),
			array( "fa fa-eye" => __( "Eye", "js_composer" ) ),
			array( "fa fa-eye-slash" => __( "Eye Slash", "js_composer" ) ),
			array( "fa fa-eyedropper" => __( "Eyedropper", "js_composer" ) ),
			array( "fa fa-fax" => __( "Fax", "js_composer" ) ),
			array( "fa fa-female" => __( "Female", "js_composer" ) ),
			array( "fa fa-fighter-jet" => __( "Fighter-jet", "js_composer" ) ),
			array( "fa fa-file-archive-o" => __( "Archive File Outlined", "js_composer" ) ),
			array( "fa fa-file-audio-o" => __( "Audio File Outlined", "js_composer" ) ),
			array( "fa fa-file-code-o" => __( "Code File Outlined", "js_composer" ) ),
			array( "fa fa-file-excel-o" => __( "Excel File Outlined", "js_composer" ) ),
			array( "fa fa-file-image-o" => __( "Image File Outlined", "js_composer" ) ),
			array( "fa fa-file-pdf-o" => __( "PDF File Outlined", "js_composer" ) ),
			array( "fa fa-file-powerpoint-o" => __( "Powerpoint File Outlined", "js_composer" ) ),
			array( "fa fa-file-video-o" => __( "Video File Outlined", "js_composer" ) ),
			array( "fa fa-file-word-o" => __( "Word File Outlined", "js_composer" ) ),
			array( "fa fa-film" => __( "Film", "js_composer" ) ),
			array( "fa fa-filter" => __( "Filter", "js_composer" ) ),
			array( "fa fa-fire" => __( "Fire", "js_composer" ) ),
			array( "fa fa-fire-extinguisher" => __( "Fire-extinguisher", "js_composer" ) ),
			array( "fa fa-flag" => __( "Flag", "js_composer" ) ),
			array( "fa fa-flag-checkered" => __( "Flag-checkered", "js_composer" ) ),
			array( "fa fa-flag-o" => __( "Flag Outlined", "js_composer" ) ),
			array( "fa fa-flask" => __( "Flask", "js_composer" ) ),
			array( "fa fa-folder" => __( "Folder", "js_composer" ) ),
			array( "fa fa-folder-o" => __( "Folder Outlined", "js_composer" ) ),
			array( "fa fa-folder-open" => __( "Folder Open", "js_composer" ) ),
			array( "fa fa-folder-open-o" => __( "Folder Open Outlined", "js_composer" ) ),
			array( "fa fa-frown-o" => __( "Frown Outlined", "js_composer" ) ),
			array( "fa fa-futbol-o" => __( "Futbol Outlined", "js_composer" ) ),
			array( "fa fa-gamepad" => __( "Gamepad", "js_composer" ) ),
			array( "fa fa-gavel" => __( "Gavel", "js_composer" ) ),
			array( "fa fa-gift" => __( "Gift", "js_composer" ) ),
			array( "fa fa-glass" => __( "Glass", "js_composer" ) ),
			array( "fa fa-globe" => __( "Globe", "js_composer" ) ),
			array( "fa fa-graduation-cap" => __( "Graduation Cap", "js_composer" ) ),
			array( "fa fa-hdd-o" => __( "HDD", "js_composer" ) ),
			array( "fa fa-headphones" => __( "Headphones", "js_composer" ) ),
			array( "fa fa-heart" => __( "Heart", "js_composer" ) ),
			array( "fa fa-heart-o" => __( "Heart Outlined", "js_composer" ) ),
			array( "fa fa-history" => __( "History", "js_composer" ) ),
			array( "fa fa-home" => __( "Home", "js_composer" ) ),
			array( "fa fa-inbox" => __( "Inbox", "js_composer" ) ),
			array( "fa fa-info" => __( "Info", "js_composer" ) ),
			array( "fa fa-info-circle" => __( "Info Circle", "js_composer" ) ),
			array( "fa fa-key" => __( "Key", "js_composer" ) ),
			array( "fa fa-keyboard-o" => __( "Keyboard Outlined", "js_composer" ) ),
			array( "fa fa-language" => __( "Language", "js_composer" ) ),
			array( "fa fa-laptop" => __( "Laptop", "js_composer" ) ),
			array( "fa fa-leaf" => __( "Leaf", "js_composer" ) ),
			array( "fa fa-lemon-o" => __( "Lemon Outlined", "js_composer" ) ),
			array( "fa fa-level-down" => __( "Level Down", "js_composer" ) ),
			array( "fa fa-level-up" => __( "Level Up", "js_composer" ) ),
			array( "fa fa-life-ring" => __( "Life Ring", "js_composer" ) ),
			array( "fa fa-lightbulb-o" => __( "Lightbulb Outlined", "js_composer" ) ),
			array( "fa fa-line-chart" => __( "Line Chart", "js_composer" ) ),
			array( "fa fa-location-arrow" => __( "Location-arrow", "js_composer" ) ),
			array( "fa fa-lock" => __( "Lock", "js_composer" ) ),
			array( "fa fa-magic" => __( "Magic", "js_composer" ) ),
			array( "fa fa-magnet" => __( "Magnet", "js_composer" ) ),
			array( "fa fa-male" => __( "Male", "js_composer" ) ),
			array( "fa fa-map-marker" => __( "Map-marker", "js_composer" ) ),
			array( "fa fa-meh-o" => __( "Meh Outlined", "js_composer" ) ),
			array( "fa fa-microphone" => __( "Microphone", "js_composer" ) ),
			array( "fa fa-microphone-slash" => __( "Microphone Slash", "js_composer" ) ),
			array( "fa fa-minus" => __( "Minus", "js_composer" ) ),
			array( "fa fa-minus-circle" => __( "Minus Circle", "js_composer" ) ),
			array( "fa fa-minus-square" => __( "Minus Square", "js_composer" ) ),
			array( "fa fa-minus-square-o" => __( "Minus Square Outlined", "js_composer" ) ),
			array( "fa fa-mobile" => __( "Mobile Phone", "js_composer" ) ),
			array( "fa fa-money" => __( "Money", "js_composer" ) ),
			array( "fa fa-moon-o" => __( "Moon Outlined", "js_composer" ) ),
			array( "fa fa-music" => __( "Music", "js_composer" ) ),
			array( "fa fa-newspaper-o" => __( "Newspaper Outlined", "js_composer" ) ),
			array( "fa fa-paint-brush" => __( "Paint Brush", "js_composer" ) ),
			array( "fa fa-paper-plane" => __( "Paper Plane", "js_composer" ) ),
			array( "fa fa-paper-plane-o" => __( "Paper Plane Outlined", "js_composer" ) ),
			array( "fa fa-paw" => __( "Paw", "js_composer" ) ),
			array( "fa fa-pencil" => __( "Pencil", "js_composer" ) ),
			array( "fa fa-pencil-square" => __( "Pencil Square", "js_composer" ) ),
			array( "fa fa-pencil-square-o" => __( "Pencil Square Outlined", "js_composer" ) ),
			array( "fa fa-phone" => __( "Phone", "js_composer" ) ),
			array( "fa fa-phone-square" => __( "Phone Square", "js_composer" ) ),
			array( "fa fa-picture-o" => __( "Picture Outlined", "js_composer" ) ),
			array( "fa fa-pie-chart" => __( "Pie Chart", "js_composer" ) ),
			array( "fa fa-plane" => __( "Plane", "js_composer" ) ),
			array( "fa fa-plug" => __( "Plug", "js_composer" ) ),
			array( "fa fa-plus" => __( "Plus", "js_composer" ) ),
			array( "fa fa-plus-circle" => __( "Plus Circle", "js_composer" ) ),
			array( "fa fa-plus-square" => __( "Plus Square", "js_composer" ) ),
			array( "fa fa-plus-square-o" => __( "Plus Square Outlined", "js_composer" ) ),
			array( "fa fa-power-off" => __( "Power Off", "js_composer" ) ),
			array( "fa fa-print" => __( "Print", "js_composer" ) ),
			array( "fa fa-puzzle-piece" => __( "Puzzle Piece", "js_composer" ) ),
			array( "fa fa-qrcode" => __( "QRcode", "js_composer" ) ),
			array( "fa fa-question" => __( "Question", "js_composer" ) ),
			array( "fa fa-question-circle" => __( "Question Circle", "js_composer" ) ),
			array( "fa fa-quote-left" => __( "Quote-left", "js_composer" ) ),
			array( "fa fa-quote-right" => __( "Quote-right", "js_composer" ) ),
			array( "fa fa-random" => __( "Random", "js_composer" ) ),
			array( "fa fa-recycle" => __( "Recycle", "js_composer" ) ),
			array( "fa fa-refresh" => __( "Refresh", "js_composer" ) ),
			array( "fa fa-reply" => __( "Reply", "js_composer" ) ),
			array( "fa fa-reply-all" => __( "Reply-all", "js_composer" ) ),
			array( "fa fa-retweet" => __( "Retweet", "js_composer" ) ),
			array( "fa fa-road" => __( "Road", "js_composer" ) ),
			array( "fa fa-rocket" => __( "Rocket", "js_composer" ) ),
			array( "fa fa-rss" => __( "RSS", "js_composer" ) ),
			array( "fa fa-rss-square" => __( "RSS Square", "js_composer" ) ),
			array( "fa fa-search" => __( "Search", "js_composer" ) ),
			array( "fa fa-search-minus" => __( "Search Minus", "js_composer" ) ),
			array( "fa fa-search-plus" => __( "Search Plus", "js_composer" ) ),
			array( "fa fa-share" => __( "Share", "js_composer" ) ),
			array( "fa fa-share-alt" => __( "Share Alt", "js_composer" ) ),
			array( "fa fa-share-alt-square" => __( "Share Alt Square", "js_composer" ) ),
			array( "fa fa-share-square" => __( "Share Square", "js_composer" ) ),
			array( "fa fa-share-square-o" => __( "Share Square Outlined", "js_composer" ) ),
			array( "fa fa-shield" => __( "shield", "js_composer" ) ),
			array( "fa fa-shopping-cart" => __( "shopping-cart", "js_composer" ) ),
			array( "fa fa-sign-in" => __( "Sign In", "js_composer" ) ),
			array( "fa fa-sign-out" => __( "Sign Out", "js_composer" ) ),
			array( "fa fa-signal" => __( "signal", "js_composer" ) ),
			array( "fa fa-sitemap" => __( "Sitemap", "js_composer" ) ),
			array( "fa fa-sliders" => __( "Sliders", "js_composer" ) ),
			array( "fa fa-smile-o" => __( "Smile Outlined", "js_composer" ) ),
			array( "fa fa-sort" => __( "Sort", "js_composer" ) ),
			array( "fa fa-sort-alpha-asc" => __( "Sort Alpha Ascending", "js_composer" ) ),
			array( "fa fa-sort-alpha-desc" => __( "Sort Alpha Descending", "js_composer" ) ),
			array( "fa fa-sort-amount-asc" => __( "Sort Amount Ascending", "js_composer" ) ),
			array( "fa fa-sort-amount-desc" => __( "Sort Amount Descending", "js_composer" ) ),
			array( "fa fa-sort-asc" => __( "Sort Ascending", "js_composer" ) ),
			array( "fa fa-sort-desc" => __( "Sort Descending", "js_composer" ) ),
			array( "fa fa-sort-numeric-asc" => __( "Sort Numeric Ascending", "js_composer" ) ),
			array( "fa fa-sort-numeric-desc" => __( "Sort Numeric Descending", "js_composer" ) ),
			array( "fa fa-space-shuttle" => __( "Space Shuttle", "js_composer" ) ),
			array( "fa fa-spinner" => __( "Spinner", "js_composer" ) ),
			array( "fa fa-spoon" => __( "spoon", "js_composer" ) ),
			array( "fa fa-square" => __( "Square", "js_composer" ) ),
			array( "fa fa-square-o" => __( "Square Outlined", "js_composer" ) ),
			array( "fa fa-star" => __( "Star", "js_composer" ) ),
			array( "fa fa-star-half" => __( "star-half", "js_composer" ) ),
			array( "fa fa-star-half-o" => __( "Star Half Outlined", "js_composer" ) ),
			array( "fa fa-star-o" => __( "Star Outlined", "js_composer" ) ),
			array( "fa fa-suitcase" => __( "Suitcase", "js_composer" ) ),
			array( "fa fa-sun-o" => __( "Sun Outlined", "js_composer" ) ),
			array( "fa fa-tablet" => __( "tablet", "js_composer" ) ),
			array( "fa fa-tachometer" => __( "Tachometer", "js_composer" ) ),
			array( "fa fa-tag" => __( "tag", "js_composer" ) ),
			array( "fa fa-tags" => __( "tags", "js_composer" ) ),
			array( "fa fa-tasks" => __( "Tasks", "js_composer" ) ),
			array( "fa fa-taxi" => __( "Taxi", "js_composer" ) ),
			array( "fa fa-terminal" => __( "Terminal", "js_composer" ) ),
			array( "fa fa-thumb-tack" => __( "Thumb Tack", "js_composer" ) ),
			array( "fa fa-thumbs-down" => __( "thumbs-down", "js_composer" ) ),
			array( "fa fa-thumbs-o-down" => __( "Thumbs Down Outlined", "js_composer" ) ),
			array( "fa fa-thumbs-o-up" => __( "Thumbs Up Outlined", "js_composer" ) ),
			array( "fa fa-thumbs-up" => __( "thumbs-up", "js_composer" ) ),
			array( "fa fa-ticket" => __( "Ticket", "js_composer" ) ),
			array( "fa fa-times" => __( "Times", "js_composer" ) ),
			array( "fa fa-times-circle" => __( "Times Circle", "js_composer" ) ),
			array( "fa fa-times-circle-o" => __( "Times Circle Outlined", "js_composer" ) ),
			array( "fa fa-tint" => __( "tint", "js_composer" ) ),
			array( "fa fa-toggle-off" => __( "Toggle Off", "js_composer" ) ),
			array( "fa fa-toggle-on" => __( "Toggle On", "js_composer" ) ),
			array( "fa fa-trash" => __( "Trash", "js_composer" ) ),
			array( "fa fa-trash-o" => __( "Trash Outlined", "js_composer" ) ),
			array( "fa fa-tree" => __( "Tree", "js_composer" ) ),
			array( "fa fa-trophy" => __( "trophy", "js_composer" ) ),
			array( "fa fa-truck" => __( "truck", "js_composer" ) ),
			array( "fa fa-tty" => __( "TTY", "js_composer" ) ),
			array( "fa fa-umbrella" => __( "Umbrella", "js_composer" ) ),
			array( "fa fa-university" => __( "University", "js_composer" ) ),
			array( "fa fa-unlock" => __( "unlock", "js_composer" ) ),
			array( "fa fa-unlock-alt" => __( "Unlock Alt", "js_composer" ) ),
			array( "fa fa-upload" => __( "Upload", "js_composer" ) ),
			array( "fa fa-user" => __( "User", "js_composer" ) ),
			array( "fa fa-users" => __( "Users", "js_composer" ) ),
			array( "fa fa-video-camera" => __( "Video Camera", "js_composer" ) ),
			array( "fa fa-volume-down" => __( "volume-down", "js_composer" ) ),
			array( "fa fa-volume-off" => __( "volume-off", "js_composer" ) ),
			array( "fa fa-volume-up" => __( "volume-up", "js_composer" ) ),
			array( "fa fa-wheelchair" => __( "Wheelchair", "js_composer" ) ),
			array( "fa fa-wifi" => __( "WiFi", "js_composer" ) ),
			array( "fa fa-wrench" => __( "Wrench", "js_composer" ) ),
		),
		"File Type Icons" => array(
			array( "fa fa-file" => __( "File", "js_composer" ) ),
			array( "fa fa-file-archive-o" => __( "Archive File Outlined", "js_composer" ) ),
			array( "fa fa-file-audio-o" => __( "Audio File Outlined", "js_composer" ) ),
			array( "fa fa-file-code-o" => __( "Code File Outlined", "js_composer" ) ),
			array( "fa fa-file-excel-o" => __( "Excel File Outlined", "js_composer" ) ),
			array( "fa fa-file-image-o" => __( "Image File Outlined", "js_composer" ) ),
			array( "fa fa-file-o" => __( "File Outlined", "js_composer" ) ),
			array( "fa fa-file-pdf-o" => __( "PDF File Outlined", "js_composer" ) ),
			array( "fa fa-file-powerpoint-o" => __( "Powerpoint File Outlined", "js_composer" ) ),
			array( "fa fa-file-text" => __( "File Text", "js_composer" ) ),
			array( "fa fa-file-text-o" => __( "File Text Outlined", "js_composer" ) ),
			array( "fa fa-file-video-o" => __( "Video File Outlined", "js_composer" ) ),
			array( "fa fa-file-word-o" => __( "Word File Outlined", "js_composer" ) ),
		),
		"Spinner Icons" => array(
			array( "fa fa-circle-o-notch" => __( "Circle Outlined Notched", "js_composer" ) ),
			array( "fa fa-cog" => __( "cog", "js_composer" ) ),
			array( "fa fa-refresh" => __( "refresh", "js_composer" ) ),
			array( "fa fa-spinner" => __( "Spinner", "js_composer" ) ),
		),
		"Form Control Icons" => array(
			array( "fa fa-check-square" => __( "Check Square", "js_composer" ) ),
			array( "fa fa-check-square-o" => __( "Check Square Outlined", "js_composer" ) ),
			array( "fa fa-circle" => __( "Circle", "js_composer" ) ),
			array( "fa fa-circle-o" => __( "Circle Outlined", "js_composer" ) ),
			array( "fa fa-dot-circle-o" => __( "Dot Circle O", "js_composer" ) ),
			array( "fa fa-minus-square" => __( "Minus Square", "js_composer" ) ),
			array( "fa fa-minus-square-o" => __( "Minus Square Outlined", "js_composer" ) ),
			array( "fa fa-plus-square" => __( "Plus Square", "js_composer" ) ),
			array( "fa fa-plus-square-o" => __( "Plus Square Outlined", "js_composer" ) ),
			array( "fa fa-square" => __( "Square", "js_composer" ) ),
			array( "fa fa-square-o" => __( "Square Outlined", "js_composer" ) ),
		),
		"Payment Icons" => array(
			array( "fa fa-cc-amex" => __( "American Express Credit Card", "js_composer" ) ),
			array( "fa fa-cc-discover" => __( "Discover Credit Card", "js_composer" ) ),
			array( "fa fa-cc-mastercard" => __( "MasterCard Credit Card", "js_composer" ) ),
			array( "fa fa-cc-paypal" => __( "Paypal Credit Card", "js_composer" ) ),
			array( "fa fa-cc-stripe" => __( "Stripe Credit Card", "js_composer" ) ),
			array( "fa fa-cc-visa" => __( "Visa Credit Card", "js_composer" ) ),
			array( "fa fa-credit-card" => __( "credit-card", "js_composer" ) ),
			array( "fa fa-google-wallet" => __( "Goole Wallet", "js_composer" ) ),
			array( "fa fa-paypal" => __( "Paypal", "js_composer" ) ),
		),
		"Chart Icons" => array(
			array( "fa fa-area-chart" => __( "Area Chart", "js_composer" ) ),
			array( "fa fa-bar-chart" => __( "Bar Chart", "js_composer" ) ),
			array( "fa fa-line-chart" => __( "Line Chart", "js_composer" ) ),
			array( "fa fa-pie-chart" => __( "Pie Chart", "js_composer" ) ),
		),
		"Currency Icons" => array(
			array( "fa fa-btc" => __( "Bitcoin (BTC)", "js_composer" ) ),
			array( "fa fa-eur" => __( "Euro (EUR)", "js_composer" ) ),
			array( "fa fa-gbp" => __( "GBP", "js_composer" ) ),
			array( "fa fa-ils" => __( "Shekel (ILS)", "js_composer" ) ),
			array( "fa fa-inr" => __( "Indian Rupee (INR)", "js_composer" ) ),
			array( "fa fa-jpy" => __( "Japanese Yen (JPY)", "js_composer" ) ),
			array( "fa fa-krw" => __( "Korean Won (KRW)", "js_composer" ) ),
			array( "fa fa-money" => __( "Money", "js_composer" ) ),
			array( "fa fa-rub" => __( "Russian Ruble (RUB)", "js_composer" ) ),
			array( "fa fa-try" => __( "Turkish Lira (TRY)", "js_composer" ) ),
			array( "fa fa-usd" => __( "US Dollar", "js_composer" ) ),
		),
		"Text Editor Icons" => array(
			array( "fa fa-align-center" => __( "Align Center", "js_composer" ) ),
			array( "fa fa-align-justify" => __( "Align Justify", "js_composer" ) ),
			array( "fa fa-align-left" => __( "Align Left", "js_composer" ) ),
			array( "fa fa-align-right" => __( "Align Right", "js_composer" ) ),
			array( "fa fa-bold" => __( "Bold", "js_composer" ) ),
			array( "fa fa-chain-broken" => __( "Chain Broken", "js_composer" ) ),
			array( "fa fa-clipboard" => __( "Clipboard", "js_composer" ) ),
			array( "fa fa-columns" => __( "Columns", "js_composer" ) ),
			array( "fa fa-eraser" => __( "Eraser", "js_composer" ) ),
			array( "fa fa-file" => __( "File", "js_composer" ) ),
			array( "fa fa-file-o" => __( "File Outlined", "js_composer" ) ),
			array( "fa fa-file-text" => __( "File Text", "js_composer" ) ),
			array( "fa fa-file-text-o" => __( "File Text Outlined", "js_composer" ) ),
			array( "fa fa-files-o" => __( "Files Outlined", "js_composer" ) ),
			array( "fa fa-floppy-o" => __( "Floppy Outlined", "js_composer" ) ),
			array( "fa fa-font" => __( "Font", "js_composer" ) ),
			array( "fa fa-header" => __( "Header", "js_composer" ) ),
			array( "fa fa-indent" => __( "Indent", "js_composer" ) ),
			array( "fa fa-italic" => __( "Italic", "js_composer" ) ),
			array( "fa fa-link" => __( "Link", "js_composer" ) ),
			array( "fa fa-list" => __( "List", "js_composer" ) ),
			array( "fa fa-list-alt" => __( "List alt", "js_composer" ) ),
			array( "fa fa-list-ol" => __( "List ol", "js_composer" ) ),
			array( "fa fa-list-ul" => __( "List ul", "js_composer" ) ),
			array( "fa fa-outdent" => __( "Outdent", "js_composer" ) ),
			array( "fa fa-paperclip" => __( "Paperclip", "js_composer" ) ),
			array( "fa fa-paragraph" => __( "paragraph", "js_composer" ) ),
			array( "fa fa-repeat" => __( "Repeat", "js_composer" ) ),
			array( "fa fa-scissors" => __( "Scissors", "js_composer" ) ),
			array( "fa fa-strikethrough" => __( "Strikethrough", "js_composer" ) ),
			array( "fa fa-subscript" => __( "Subscript", "js_composer" ) ),
			array( "fa fa-superscript" => __( "Superscript", "js_composer" ) ),
			array( "fa fa-table" => __( "Table", "js_composer" ) ),
			array( "fa fa-text-height" => __( "Text-height", "js_composer" ) ),
			array( "fa fa-text-width" => __( "Text-width", "js_composer" ) ),
			array( "fa fa-th" => __( "Th", "js_composer" ) ),
			array( "fa fa-th-large" => __( "Th-large", "js_composer" ) ),
			array( "fa fa-th-list" => __( "Th-list", "js_composer" ) ),
			array( "fa fa-underline" => __( "Underline", "js_composer" ) ),
			array( "fa fa-undo" => __( "Undo", "js_composer" ) ),
		),
		"Directional Icons" => array(
			array( "fa fa-angle-double-down" => __( "Angle Double Down", "js_composer" ) ),
			array( "fa fa-angle-double-left" => __( "Angle Double Left", "js_composer" ) ),
			array( "fa fa-angle-double-right" => __( "Angle Double Right", "js_composer" ) ),
			array( "fa fa-angle-double-up" => __( "Angle Double Up", "js_composer" ) ),
			array( "fa fa-angle-down" => __( "Angle Down", "js_composer" ) ),
			array( "fa fa-angle-left" => __( "Angle Left", "js_composer" ) ),
			array( "fa fa-angle-right" => __( "Angle Right", "js_composer" ) ),
			array( "fa fa-angle-up" => __( "Angle Up", "js_composer" ) ),
			array( "fa fa-arrow-circle-down" => __( "Arrow Circle Down", "js_composer" ) ),
			array( "fa fa-arrow-circle-left" => __( "Arrow Circle Left", "js_composer" ) ),
			array( "fa fa-arrow-circle-o-down" => __( "Arrow Circle Outlined Down", "js_composer" ) ),
			array( "fa fa-arrow-circle-o-left" => __( "Arrow Circle Outlined Left", "js_composer" ) ),
			array( "fa fa-arrow-circle-o-right" => __( "Arrow Circle Outlined Right", "js_composer" ) ),
			array( "fa fa-arrow-circle-o-up" => __( "Arrow Circle Outlined Up", "js_composer" ) ),
			array( "fa fa-arrow-circle-right" => __( "Arrow Circle Right", "js_composer" ) ),
			array( "fa fa-arrow-circle-up" => __( "Arrow Circle Up", "js_composer" ) ),
			array( "fa fa-arrow-down" => __( "Arrow Down", "js_composer" ) ),
			array( "fa fa-arrow-left" => __( "Arrow Left", "js_composer" ) ),
			array( "fa fa-arrow-right" => __( "Arrow Right", "js_composer" ) ),
			array( "fa fa-arrow-up" => __( "Arrow Up", "js_composer" ) ),
			array( "fa fa-arrows" => __( "Arrows", "js_composer" ) ),
			array( "fa fa-arrows-alt" => __( "Arrows Alt", "js_composer" ) ),
			array( "fa fa-arrows-h" => __( "Arrows Horizontal", "js_composer" ) ),
			array( "fa fa-arrows-v" => __( "Arrows Vertical", "js_composer" ) ),
			array( "fa fa-caret-down" => __( "Caret Down", "js_composer" ) ),
			array( "fa fa-caret-left" => __( "Caret Left", "js_composer" ) ),
			array( "fa fa-caret-right" => __( "Caret Right", "js_composer" ) ),
			array( "fa fa-caret-square-o-down" => __( "Caret Square Outlined Down", "js_composer" ) ),
			array( "fa fa-caret-square-o-left" => __( "Caret Square Outlined Left", "js_composer" ) ),
			array( "fa fa-caret-square-o-right" => __( "Caret Square Outlined Right", "js_composer" ) ),
			array( "fa fa-caret-square-o-up" => __( "Caret Square Outlined Up", "js_composer" ) ),
			array( "fa fa-caret-up" => __( "Caret Up", "js_composer" ) ),
			array( "fa fa-chevron-circle-down" => __( "Chevron Circle Down", "js_composer" ) ),
			array( "fa fa-chevron-circle-left" => __( "Chevron Circle Left", "js_composer" ) ),
			array( "fa fa-chevron-circle-right" => __( "Chevron Circle Right", "js_composer" ) ),
			array( "fa fa-chevron-circle-up" => __( "Chevron Circle Up", "js_composer" ) ),
			array( "fa fa-chevron-down" => __( "Chevron Down", "js_composer" ) ),
			array( "fa fa-chevron-left" => __( "Chevron Left", "js_composer" ) ),
			array( "fa fa-chevron-right" => __( "Chevron Right", "js_composer" ) ),
			array( "fa fa-chevron-up" => __( "Chevron Up", "js_composer" ) ),
			array( "fa fa-hand-o-down" => __( "Hand Outlined Down", "js_composer" ) ),
			array( "fa fa-hand-o-left" => __( "Hand Outlined Left", "js_composer" ) ),
			array( "fa fa-hand-o-right" => __( "Hand Outlined Right", "js_composer" ) ),
			array( "fa fa-hand-o-up" => __( "Hand Outlined Up", "js_composer" ) ),
			array( "fa fa-long-arrow-down" => __( "Long Arrow Down", "js_composer" ) ),
			array( "fa fa-long-arrow-left" => __( "Long Arrow Left", "js_composer" ) ),
			array( "fa fa-long-arrow-right" => __( "Long Arrow Right", "js_composer" ) ),
			array( "fa fa-long-arrow-up" => __( "Long Arrow Up", "js_composer" ) ),
		),
		"Video Player Icons" => array(
			array( "fa fa-arrows-alt" => __( "Arrows Alt", "js_composer" ) ),
			array( "fa fa-backward" => __( "Backward", "js_composer" ) ),
			array( "fa fa-compress" => __( "Compress", "js_composer" ) ),
			array( "fa fa-eject" => __( "Eject", "js_composer" ) ),
			array( "fa fa-expand" => __( "Expand", "js_composer" ) ),
			array( "fa fa-fast-backward" => __( "Fast Backward", "js_composer" ) ),
			array( "fa fa-fast-forward" => __( "Fast Forward", "js_composer" ) ),
			array( "fa fa-forward" => __( "Forward", "js_composer" ) ),
			array( "fa fa-pause" => __( "Pause", "js_composer" ) ),
			array( "fa fa-play" => __( "Play", "js_composer" ) ),
			array( "fa fa-play-circle" => __( "Play Circle", "js_composer" ) ),
			array( "fa fa-play-circle-o" => __( "Play Circle Outlined", "js_composer" ) ),
			array( "fa fa-step-backward" => __( "Step Backward", "js_composer" ) ),
			array( "fa fa-step-forward" => __( "Step Forward", "js_composer" ) ),
			array( "fa fa-stop" => __( "Stop", "js_composer" ) ),
			array( "fa fa-youtube-play" => __( "YouTube Play", "js_composer" ) ),
		),
		"Brand Icons" => array(
			array( "fa fa-adn" => __( "App.net", "js_composer" ) ),
			array( "fa fa-android" => __( "Android", "js_composer" ) ),
			array( "fa fa-angellist" => __( "AngelList", "js_composer" ) ),
			array( "fa fa-apple" => __( "Apple", "js_composer" ) ),
			array( "fa fa-behance" => __( "Behance", "js_composer" ) ),
			array( "fa fa-behance-square" => __( "Behance Square", "js_composer" ) ),
			array( "fa fa-bitbucket" => __( "Bitbucket", "js_composer" ) ),
			array( "fa fa-bitbucket-square" => __( "Bitbucket Square", "js_composer" ) ),
			array( "fa fa-btc" => __( "Bitcoin (BTC)", "js_composer" ) ),
			array( "fa fa-cc-amex" => __( "American Express Credit Card", "js_composer" ) ),
			array( "fa fa-cc-discover" => __( "Discover Credit Card", "js_composer" ) ),
			array( "fa fa-cc-mastercard" => __( "MasterCard Credit Card", "js_composer" ) ),
			array( "fa fa-cc-paypal" => __( "Paypal Credit Card", "js_composer" ) ),
			array( "fa fa-cc-stripe" => __( "Stripe Credit Card", "js_composer" ) ),
			array( "fa fa-cc-visa" => __( "Visa Credit Card", "js_composer" ) ),
			array( "fa fa-codepen" => __( "Codepen", "js_composer" ) ),
			array( "fa fa-css3" => __( "CSS 3 Logo", "js_composer" ) ),
			array( "fa fa-delicious" => __( "Delicious Logo", "js_composer" ) ),
			array( "fa fa-deviantart" => __( "deviantART", "js_composer" ) ),
			array( "fa fa-digg" => __( "Digg Logo", "js_composer" ) ),
			array( "fa fa-dribbble" => __( "Dribbble", "js_composer" ) ),
			array( "fa fa-dropbox" => __( "Dropbox", "js_composer" ) ),
			array( "fa fa-drupal" => __( "Drupal Logo", "js_composer" ) ),
			array( "fa fa-empire" => __( "Galactic Empire", "js_composer" ) ),
			array( "fa fa-facebook" => __( "Facebook", "js_composer" ) ),
			array( "fa fa-facebook-square" => __( "Facebook Square", "js_composer" ) ),
			array( "fa fa-flickr" => __( "Flickr", "js_composer" ) ),
			array( "fa fa-foursquare" => __( "Foursquare", "js_composer" ) ),
			array( "fa fa-git" => __( "Git", "js_composer" ) ),
			array( "fa fa-git-square" => __( "Git Square", "js_composer" ) ),
			array( "fa fa-github" => __( "GitHub", "js_composer" ) ),
			array( "fa fa-github-alt" => __( "GitHub Alt", "js_composer" ) ),
			array( "fa fa-github-square" => __( "GitHub Square", "js_composer" ) ),
			array( "fa fa-gittip" => __( "Gittip", "js_composer" ) ),
			array( "fa fa-google" => __( "Google Logo", "js_composer" ) ),
			array( "fa fa-google-plus" => __( "Google Plus", "js_composer" ) ),
			array( "fa fa-google-plus-square" => __( "Google Plus Square", "js_composer" ) ),
			array( "fa fa-google-wallet" => __( "Goole Wallet", "js_composer" ) ),
			array( "fa fa-hacker-news" => __( "Hacker News", "js_composer" ) ),
			array( "fa fa-html5" => __( "HTML 5 Logo", "js_composer" ) ),
			array( "fa fa-instagram" => __( "Instagram", "js_composer" ) ),
			array( "fa fa-ioxhost" => __( "ioxhost", "js_composer" ) ),
			array( "fa fa-joomla" => __( "Joomla Logo", "js_composer" ) ),
			array( "fa fa-jsfiddle" => __( "jsFiddle", "js_composer" ) ),
			array( "fa fa-lastfm" => __( "last.fm", "js_composer" ) ),
			array( "fa fa-lastfm-square" => __( "last.fm Square", "js_composer" ) ),
			array( "fa fa-linkedin" => __( "LinkedIn", "js_composer" ) ),
			array( "fa fa-linkedin-square" => __( "LinkedIn Square", "js_composer" ) ),
			array( "fa fa-linux" => __( "Linux", "js_composer" ) ),
			array( "fa fa-maxcdn" => __( "MaxCDN", "js_composer" ) ),
			array( "fa fa-meanpath" => __( "meanpath", "js_composer" ) ),
			array( "fa fa-openid" => __( "OpenID", "js_composer" ) ),
			array( "fa fa-pagelines" => __( "Pagelines", "js_composer" ) ),
			array( "fa fa-paypal" => __( "Paypal", "js_composer" ) ),
			array( "fa fa-pied-piper" => __( "Pied Piper Logo", "js_composer" ) ),
			array( "fa fa-pied-piper-alt" => __( "Pied Piper Alternate Logo", "js_composer" ) ),
			array( "fa fa-pinterest" => __( "Pinterest", "js_composer" ) ),
			array( "fa fa-pinterest-square" => __( "Pinterest Square", "js_composer" ) ),
			array( "fa fa-qq" => __( "QQ", "js_composer" ) ),
			array( "fa fa-rebel" => __( "Rebel Alliance", "js_composer" ) ),
			array( "fa fa-reddit" => __( "reddit Logo", "js_composer" ) ),
			array( "fa fa-reddit-square" => __( "reddit Square", "js_composer" ) ),
			array( "fa fa-renren" => __( "Renren", "js_composer" ) ),
			array( "fa fa-share-alt" => __( "Share Alt", "js_composer" ) ),
			array( "fa fa-share-alt-square" => __( "Share Alt Square", "js_composer" ) ),
			array( "fa fa-skype" => __( "Skype", "js_composer" ) ),
			array( "fa fa-slack" => __( "Slack Logo", "js_composer" ) ),
			array( "fa fa-slideshare" => __( "Slideshare", "js_composer" ) ),
			array( "fa fa-soundcloud" => __( "SoundCloud", "js_composer" ) ),
			array( "fa fa-spotify" => __( "Spotify", "js_composer" ) ),
			array( "fa fa-stack-exchange" => __( "Stack Exchange", "js_composer" ) ),
			array( "fa fa-stack-overflow" => __( "Stack Overflow", "js_composer" ) ),
			array( "fa fa-steam" => __( "Steam", "js_composer" ) ),
			array( "fa fa-steam-square" => __( "Steam Square", "js_composer" ) ),
			array( "fa fa-stumbleupon" => __( "StumbleUpon Logo", "js_composer" ) ),
			array( "fa fa-stumbleupon-circle" => __( "StumbleUpon Circle", "js_composer" ) ),
			array( "fa fa-tencent-weibo" => __( "Tencent Weibo", "js_composer" ) ),
			array( "fa fa-trello" => __( "Trello", "js_composer" ) ),
			array( "fa fa-tumblr" => __( "Tumblr", "js_composer" ) ),
			array( "fa fa-tumblr-square" => __( "Tumblr Square", "js_composer" ) ),
			array( "fa fa-twitch" => __( "Twitch", "js_composer" ) ),
			array( "fa fa-twitter" => __( "Twitter", "js_composer" ) ),
			array( "fa fa-twitter-square" => __( "Twitter Square", "js_composer" ) ),
			array( "fa fa-vimeo-square" => __( "Vimeo Square", "js_composer" ) ),
			array( "fa fa-vine" => __( "Vine", "js_composer" ) ),
			array( "fa fa-vk" => __( "VK", "js_composer" ) ),
			array( "fa fa-weibo" => __( "Weibo", "js_composer" ) ),
			array( "fa fa-weixin" => __( "Weixin (WeChat)", "js_composer" ) ),
			array( "fa fa-windows" => __( "Windows", "js_composer" ) ),
			array( "fa fa-wordpress" => __( "Wordpress Logo", "js_composer" ) ),
			array( "fa fa-xing" => __( "Xing", "js_composer" ) ),
			array( "fa fa-xing-square" => __( "Xing Square", "js_composer" ) ),
			array( "fa fa-yahoo" => __( "Yahoo Logo", "js_composer" ) ),
			array( "fa fa-yelp" => __( "Yelp", "js_composer" ) ),
			array( "fa fa-youtube" => __( "YouTube", "js_composer" ) ),
			array( "fa fa-youtube-play" => __( "YouTube Play", "js_composer" ) ),
			array( "fa fa-youtube-square" => __( "YouTube Square", "js_composer" ) ),
		),
		"Medical Icons" => array(
			array( "fa fa-ambulance" => __( "Ambulance", "js_composer" ) ),
			array( "fa fa-h-square" => __( "H Square", "js_composer" ) ),
			array( "fa fa-hospital-o" => __( "Hospital Outlined", "js_composer" ) ),
			array( "fa fa-medkit" => __( "Medkit", "js_composer" ) ),
			array( "fa fa-plus-square" => __( "Plus Square", "js_composer" ) ),
			array( "fa fa-stethoscope" => __( "Stethoscope", "js_composer" ) ),
			array( "fa fa-user-md" => __( "User-md", "js_composer" ) ),
			array( "fa fa-wheelchair" => __( "Wheelchair", "js_composer" ) ),
		)
	);

	return array_merge( $icons, $fontawesome_icons );
}

add_filter( 'vc_iconpicker-type-openiconic', 'vc_iconpicker_type_openiconic' );

/**
 * Openicons icons from fontello.com
 *
 * @param $icons - taken from filter - vc_map param field settings['source'] provided icons (default empty array).
 * If array categorized it will auto-enable category dropdown
 *
 * @since 4.4
 * @return array - of icons for iconpicker, can be categorized, or not.
 */
function vc_iconpicker_type_openiconic( $icons ) {
	$openiconic_icons = array(
		array( "vc-oi vc-oi-dial" => __( "Dial", "js_composer" ) ),
		array( "vc-oi vc-oi-pilcrow" => __( "Pilcrow", "js_composer" ) ),
		array( "vc-oi vc-oi-at" => __( "At", "js_composer" ) ),
		array( "vc-oi vc-oi-hash" => __( "Hash", "js_composer" ) ),
		array( "vc-oi vc-oi-key-inv" => __( "Key-inv", "js_composer" ) ),
		array( "vc-oi vc-oi-key" => __( "Key", "js_composer" ) ),
		array( "vc-oi vc-oi-chart-pie-alt" => __( "Chart-pie-alt", "js_composer" ) ),
		array( "vc-oi vc-oi-chart-pie" => __( "Chart-pie", "js_composer" ) ),
		array( "vc-oi vc-oi-chart-bar" => __( "Chart-bar", "js_composer" ) ),
		array( "vc-oi vc-oi-umbrella" => __( "Umbrella", "js_composer" ) ),
		array( "vc-oi vc-oi-moon-inv" => __( "Moon-inv", "js_composer" ) ),
		array( "vc-oi vc-oi-mobile" => __( "Mobile", "js_composer" ) ),
		array( "vc-oi vc-oi-cd" => __( "Cd", "js_composer" ) ),
		array( "vc-oi vc-oi-split" => __( "Split", "js_composer" ) ),
		array( "vc-oi vc-oi-exchange" => __( "Exchange", "js_composer" ) ),
		array( "vc-oi vc-oi-block" => __( "Block", "js_composer" ) ),
		array( "vc-oi vc-oi-resize-full" => __( "Resize-full", "js_composer" ) ),
		array( "vc-oi vc-oi-article-alt" => __( "Article-alt", "js_composer" ) ),
		array( "vc-oi vc-oi-article" => __( "Article", "js_composer" ) ),
		array( "vc-oi vc-oi-pencil-alt" => __( "Pencil-alt", "js_composer" ) ),
		array( "vc-oi vc-oi-undo" => __( "Undo", "js_composer" ) ),
		array( "vc-oi vc-oi-attach" => __( "Attach", "js_composer" ) ),
		array( "vc-oi vc-oi-link" => __( "Link", "js_composer" ) ),
		array( "vc-oi vc-oi-search" => __( "Search", "js_composer" ) ),
		array( "vc-oi vc-oi-mail" => __( "Mail", "js_composer" ) ),
		array( "vc-oi vc-oi-heart" => __( "Heart", "js_composer" ) ),
		array( "vc-oi vc-oi-comment" => __( "Comment", "js_composer" ) ),
		array( "vc-oi vc-oi-resize-full-alt" => __( "Resize-full-alt", "js_composer" ) ),
		array( "vc-oi vc-oi-lock" => __( "Lock", "js_composer" ) ),
		array( "vc-oi vc-oi-book-open" => __( "Book-open", "js_composer" ) ),
		array( "vc-oi vc-oi-arrow-curved" => __( "Arrow-curved", "js_composer" ) ),
		array( "vc-oi vc-oi-equalizer" => __( "Equalizer", "js_composer" ) ),
		array( "vc-oi vc-oi-heart-empty" => __( "Heart-empty", "js_composer" ) ),
		array( "vc-oi vc-oi-lock-empty" => __( "Lock-empty", "js_composer" ) ),
		array( "vc-oi vc-oi-comment-inv" => __( "Comment-inv", "js_composer" ) ),
		array( "vc-oi vc-oi-folder" => __( "Folder", "js_composer" ) ),
		array( "vc-oi vc-oi-resize-small" => __( "Resize-small", "js_composer" ) ),
		array( "vc-oi vc-oi-play" => __( "Play", "js_composer" ) ),
		array( "vc-oi vc-oi-cursor" => __( "Cursor", "js_composer" ) ),
		array( "vc-oi vc-oi-aperture" => __( "Aperture", "js_composer" ) ),
		array( "vc-oi vc-oi-play-circle2" => __( "Play-circle2", "js_composer" ) ),
		array( "vc-oi vc-oi-resize-small-alt" => __( "Resize-small-alt", "js_composer" ) ),
		array( "vc-oi vc-oi-folder-empty" => __( "Folder-empty", "js_composer" ) ),
		array( "vc-oi vc-oi-comment-alt" => __( "Comment-alt", "js_composer" ) ),
		array( "vc-oi vc-oi-lock-open" => __( "Lock-open", "js_composer" ) ),
		array( "vc-oi vc-oi-star" => __( "Star", "js_composer" ) ),
		array( "vc-oi vc-oi-user" => __( "User", "js_composer" ) ),
		array( "vc-oi vc-oi-lock-open-empty" => __( "Lock-open-empty", "js_composer" ) ),
		array( "vc-oi vc-oi-box" => __( "Box", "js_composer" ) ),
		array( "vc-oi vc-oi-resize-vertical" => __( "Resize-vertical", "js_composer" ) ),
		array( "vc-oi vc-oi-stop" => __( "Stop", "js_composer" ) ),
		array( "vc-oi vc-oi-aperture-alt" => __( "Aperture-alt", "js_composer" ) ),
		array( "vc-oi vc-oi-book" => __( "Book", "js_composer" ) ),
		array( "vc-oi vc-oi-steering-wheel" => __( "Steering-wheel", "js_composer" ) ),
		array( "vc-oi vc-oi-pause" => __( "Pause", "js_composer" ) ),
		array( "vc-oi vc-oi-to-start" => __( "To-start", "js_composer" ) ),
		array( "vc-oi vc-oi-move" => __( "Move", "js_composer" ) ),
		array( "vc-oi vc-oi-resize-horizontal" => __( "Resize-horizontal", "js_composer" ) ),
		array( "vc-oi vc-oi-rss-alt" => __( "Rss-alt", "js_composer" ) ),
		array( "vc-oi vc-oi-comment-alt2" => __( "Comment-alt2", "js_composer" ) ),
		array( "vc-oi vc-oi-rss" => __( "Rss", "js_composer" ) ),
		array( "vc-oi vc-oi-comment-inv-alt" => __( "Comment-inv-alt", "js_composer" ) ),
		array( "vc-oi vc-oi-comment-inv-alt2" => __( "Comment-inv-alt2", "js_composer" ) ),
		array( "vc-oi vc-oi-eye" => __( "Eye", "js_composer" ) ),
		array( "vc-oi vc-oi-pin" => __( "Pin", "js_composer" ) ),
		array( "vc-oi vc-oi-video" => __( "Video", "js_composer" ) ),
		array( "vc-oi vc-oi-picture" => __( "Picture", "js_composer" ) ),
		array( "vc-oi vc-oi-camera" => __( "Camera", "js_composer" ) ),
		array( "vc-oi vc-oi-tag" => __( "Tag", "js_composer" ) ),
		array( "vc-oi vc-oi-chat" => __( "Chat", "js_composer" ) ),
		array( "vc-oi vc-oi-cog" => __( "Cog", "js_composer" ) ),
		array( "vc-oi vc-oi-popup" => __( "Popup", "js_composer" ) ),
		array( "vc-oi vc-oi-to-end" => __( "To-end", "js_composer" ) ),
		array( "vc-oi vc-oi-book-alt" => __( "Book-alt", "js_composer" ) ),
		array( "vc-oi vc-oi-brush" => __( "Brush", "js_composer" ) ),
		array( "vc-oi vc-oi-eject" => __( "Eject", "js_composer" ) ),
		array( "vc-oi vc-oi-down" => __( "Down", "js_composer" ) ),
		array( "vc-oi vc-oi-wrench" => __( "Wrench", "js_composer" ) ),
		array( "vc-oi vc-oi-chat-inv" => __( "Chat-inv", "js_composer" ) ),
		array( "vc-oi vc-oi-tag-empty" => __( "Tag-empty", "js_composer" ) ),
		array( "vc-oi vc-oi-ok" => __( "Ok", "js_composer" ) ),
		array( "vc-oi vc-oi-ok-circle" => __( "Ok-circle", "js_composer" ) ),
		array( "vc-oi vc-oi-download" => __( "Download", "js_composer" ) ),
		array( "vc-oi vc-oi-location" => __( "Location", "js_composer" ) ),
		array( "vc-oi vc-oi-share" => __( "Share", "js_composer" ) ),
		array( "vc-oi vc-oi-left" => __( "Left", "js_composer" ) ),
		array( "vc-oi vc-oi-target" => __( "Target", "js_composer" ) ),
		array( "vc-oi vc-oi-brush-alt" => __( "Brush-alt", "js_composer" ) ),
		array( "vc-oi vc-oi-cancel" => __( "Cancel", "js_composer" ) ),
		array( "vc-oi vc-oi-upload" => __( "Upload", "js_composer" ) ),
		array( "vc-oi vc-oi-location-inv" => __( "Location-inv", "js_composer" ) ),
		array( "vc-oi vc-oi-calendar" => __( "Calendar", "js_composer" ) ),
		array( "vc-oi vc-oi-right" => __( "Right", "js_composer" ) ),
		array( "vc-oi vc-oi-signal" => __( "Signal", "js_composer" ) ),
		array( "vc-oi vc-oi-eyedropper" => __( "Eyedropper", "js_composer" ) ),
		array( "vc-oi vc-oi-layers" => __( "Layers", "js_composer" ) ),
		array( "vc-oi vc-oi-award" => __( "Award", "js_composer" ) ),
		array( "vc-oi vc-oi-up" => __( "Up", "js_composer" ) ),
		array( "vc-oi vc-oi-calendar-inv" => __( "Calendar-inv", "js_composer" ) ),
		array( "vc-oi vc-oi-location-alt" => __( "Location-alt", "js_composer" ) ),
		array( "vc-oi vc-oi-download-cloud" => __( "Download-cloud", "js_composer" ) ),
		array( "vc-oi vc-oi-cancel-circle" => __( "Cancel-circle", "js_composer" ) ),
		array( "vc-oi vc-oi-plus" => __( "Plus", "js_composer" ) ),
		array( "vc-oi vc-oi-upload-cloud" => __( "Upload-cloud", "js_composer" ) ),
		array( "vc-oi vc-oi-compass" => __( "Compass", "js_composer" ) ),
		array( "vc-oi vc-oi-calendar-alt" => __( "Calendar-alt", "js_composer" ) ),
		array( "vc-oi vc-oi-down-circle" => __( "Down-circle", "js_composer" ) ),
		array( "vc-oi vc-oi-award-empty" => __( "Award-empty", "js_composer" ) ),
		array( "vc-oi vc-oi-layers-alt" => __( "Layers-alt", "js_composer" ) ),
		array( "vc-oi vc-oi-sun" => __( "Sun", "js_composer" ) ),
		array( "vc-oi vc-oi-list" => __( "List", "js_composer" ) ),
		array( "vc-oi vc-oi-left-circle" => __( "Left-circle", "js_composer" ) ),
		array( "vc-oi vc-oi-mic" => __( "Mic", "js_composer" ) ),
		array( "vc-oi vc-oi-trash" => __( "Trash", "js_composer" ) ),
		array( "vc-oi vc-oi-quote-left" => __( "Quote-left", "js_composer" ) ),
		array( "vc-oi vc-oi-plus-circle" => __( "Plus-circle", "js_composer" ) ),
		array( "vc-oi vc-oi-minus" => __( "Minus", "js_composer" ) ),
		array( "vc-oi vc-oi-quote-right" => __( "Quote-right", "js_composer" ) ),
		array( "vc-oi vc-oi-trash-empty" => __( "Trash-empty", "js_composer" ) ),
		array( "vc-oi vc-oi-volume-off" => __( "Volume-off", "js_composer" ) ),
		array( "vc-oi vc-oi-right-circle" => __( "Right-circle", "js_composer" ) ),
		array( "vc-oi vc-oi-list-nested" => __( "List-nested", "js_composer" ) ),
		array( "vc-oi vc-oi-sun-inv" => __( "Sun-inv", "js_composer" ) ),
		array( "vc-oi vc-oi-bat-empty" => __( "Bat-empty", "js_composer" ) ),
		array( "vc-oi vc-oi-up-circle" => __( "Up-circle", "js_composer" ) ),
		array( "vc-oi vc-oi-volume-up" => __( "Volume-up", "js_composer" ) ),
		array( "vc-oi vc-oi-doc" => __( "Doc", "js_composer" ) ),
		array( "vc-oi vc-oi-quote-left-alt" => __( "Quote-left-alt", "js_composer" ) ),
		array( "vc-oi vc-oi-minus-circle" => __( "Minus-circle", "js_composer" ) ),
		array( "vc-oi vc-oi-cloud" => __( "Cloud", "js_composer" ) ),
		array( "vc-oi vc-oi-rain" => __( "Rain", "js_composer" ) ),
		array( "vc-oi vc-oi-bat-half" => __( "Bat-half", "js_composer" ) ),
		array( "vc-oi vc-oi-cw" => __( "Cw", "js_composer" ) ),
		array( "vc-oi vc-oi-headphones" => __( "Headphones", "js_composer" ) ),
		array( "vc-oi vc-oi-doc-inv" => __( "Doc-inv", "js_composer" ) ),
		array( "vc-oi vc-oi-quote-right-alt" => __( "Quote-right-alt", "js_composer" ) ),
		array( "vc-oi vc-oi-help" => __( "Help", "js_composer" ) ),
		array( "vc-oi vc-oi-info" => __( "Info", "js_composer" ) ),
		array( "vc-oi vc-oi-pencil" => __( "Pencil", "js_composer" ) ),
		array( "vc-oi vc-oi-doc-alt" => __( "Doc-alt", "js_composer" ) ),
		array( "vc-oi vc-oi-clock" => __( "Clock", "js_composer" ) ),
		array( "vc-oi vc-oi-loop" => __( "Loop", "js_composer" ) ),
		array( "vc-oi vc-oi-bat-full" => __( "Bat-full", "js_composer" ) ),
		array( "vc-oi vc-oi-flash" => __( "Flash", "js_composer" ) ),
		array( "vc-oi vc-oi-moon" => __( "Moon", "js_composer" ) ),
		array( "vc-oi vc-oi-bat-charge" => __( "Bat-charge", "js_composer" ) ),
		array( "vc-oi vc-oi-loop-alt" => __( "Loop-alt", "js_composer" ) ),
		array( "vc-oi vc-oi-lamp" => __( "Lamp", "js_composer" ) ),
		array( "vc-oi vc-oi-doc-inv-alt" => __( "Doc-inv-alt", "js_composer" ) ),
		array( "vc-oi vc-oi-pencil-neg" => __( "Pencil-neg", "js_composer" ) ),
		array( "vc-oi vc-oi-home" => __( "Home", "js_composer" ) ),
	);

	return array_merge( $icons, $openiconic_icons );
}

add_filter( 'vc_iconpicker-type-typicons', 'vc_iconpicker_type_typicons' );

/**
 * Typicons icons from github.com/stephenhutchings/typicons.font
 *
 * @param $icons - taken from filter - vc_map param field settings['source'] provided icons (default empty array).
 * If array categorized it will auto-enable category dropdown
 *
 * @since 4.4
 * @return array - of icons for iconpicker, can be categorized, or not.
 */
function vc_iconpicker_type_typicons( $icons ) {
	$typicons_icons = array(
		array( "typcn typcn-adjust-brightness" => __( "Adjust Brightness", "js_composer" ) ),
		array( "typcn typcn-adjust-contrast" => __( "Adjust Contrast", "js_composer" ) ),
		array( "typcn typcn-anchor-outline" => __( "Anchor Outline", "js_composer" ) ),
		array( "typcn typcn-anchor" => __( "Anchor", "js_composer" ) ),
		array( "typcn typcn-archive" => __( "Archive", "js_composer" ) ),
		array( "typcn typcn-arrow-back-outline" => __( "Arrow Back Outline", "js_composer" ) ),
		array( "typcn typcn-arrow-back" => __( "Arrow Back", "js_composer" ) ),
		array( "typcn typcn-arrow-down-outline" => __( "Arrow Down Outline", "js_composer" ) ),
		array( "typcn typcn-arrow-down-thick" => __( "Arrow Down Thick", "js_composer" ) ),
		array( "typcn typcn-arrow-down" => __( "Arrow Down", "js_composer" ) ),
		array( "typcn typcn-arrow-forward-outline" => __( "Arrow Forward Outline", "js_composer" ) ),
		array( "typcn typcn-arrow-forward" => __( "Arrow Forward", "js_composer" ) ),
		array( "typcn typcn-arrow-left-outline" => __( "Arrow Left Outline", "js_composer" ) ),
		array( "typcn typcn-arrow-left-thick" => __( "Arrow Left Thick", "js_composer" ) ),
		array( "typcn typcn-arrow-left" => __( "Arrow Left", "js_composer" ) ),
		array( "typcn typcn-arrow-loop-outline" => __( "Arrow Loop Outline", "js_composer" ) ),
		array( "typcn typcn-arrow-loop" => __( "Arrow Loop", "js_composer" ) ),
		array( "typcn typcn-arrow-maximise-outline" => __( "Arrow Maximise Outline", "js_composer" ) ),
		array( "typcn typcn-arrow-maximise" => __( "Arrow Maximise", "js_composer" ) ),
		array( "typcn typcn-arrow-minimise-outline" => __( "Arrow Minimise Outline", "js_composer" ) ),
		array( "typcn typcn-arrow-minimise" => __( "Arrow Minimise", "js_composer" ) ),
		array( "typcn typcn-arrow-move-outline" => __( "Arrow Move Outline", "js_composer" ) ),
		array( "typcn typcn-arrow-move" => __( "Arrow Move", "js_composer" ) ),
		array( "typcn typcn-arrow-repeat-outline" => __( "Arrow Repeat Outline", "js_composer" ) ),
		array( "typcn typcn-arrow-repeat" => __( "Arrow Repeat", "js_composer" ) ),
		array( "typcn typcn-arrow-right-outline" => __( "Arrow Right Outline", "js_composer" ) ),
		array( "typcn typcn-arrow-right-thick" => __( "Arrow Right Thick", "js_composer" ) ),
		array( "typcn typcn-arrow-right" => __( "Arrow Right", "js_composer" ) ),
		array( "typcn typcn-arrow-shuffle" => __( "Arrow Shuffle", "js_composer" ) ),
		array( "typcn typcn-arrow-sorted-down" => __( "Arrow Sorted Down", "js_composer" ) ),
		array( "typcn typcn-arrow-sorted-up" => __( "Arrow Sorted Up", "js_composer" ) ),
		array( "typcn typcn-arrow-sync-outline" => __( "Arrow Sync Outline", "js_composer" ) ),
		array( "typcn typcn-arrow-sync" => __( "Arrow Sync", "js_composer" ) ),
		array( "typcn typcn-arrow-unsorted" => __( "Arrow Unsorted", "js_composer" ) ),
		array( "typcn typcn-arrow-up-outline" => __( "Arrow Up Outline", "js_composer" ) ),
		array( "typcn typcn-arrow-up-thick" => __( "Arrow Up Thick", "js_composer" ) ),
		array( "typcn typcn-arrow-up" => __( "Arrow Up", "js_composer" ) ),
		array( "typcn typcn-at" => __( "At", "js_composer" ) ),
		array( "typcn typcn-attachment-outline" => __( "Attachment Outline", "js_composer" ) ),
		array( "typcn typcn-attachment" => __( "Attachment", "js_composer" ) ),
		array( "typcn typcn-backspace-outline" => __( "Backspace Outline", "js_composer" ) ),
		array( "typcn typcn-backspace" => __( "Backspace", "js_composer" ) ),
		array( "typcn typcn-battery-charge" => __( "Battery Charge", "js_composer" ) ),
		array( "typcn typcn-battery-full" => __( "Battery Full", "js_composer" ) ),
		array( "typcn typcn-battery-high" => __( "Battery High", "js_composer" ) ),
		array( "typcn typcn-battery-low" => __( "Battery Low", "js_composer" ) ),
		array( "typcn typcn-battery-mid" => __( "Battery Mid", "js_composer" ) ),
		array( "typcn typcn-beaker" => __( "Beaker", "js_composer" ) ),
		array( "typcn typcn-beer" => __( "Beer", "js_composer" ) ),
		array( "typcn typcn-bell" => __( "Bell", "js_composer" ) ),
		array( "typcn typcn-book" => __( "Book", "js_composer" ) ),
		array( "typcn typcn-bookmark" => __( "Bookmark", "js_composer" ) ),
		array( "typcn typcn-briefcase" => __( "Briefcase", "js_composer" ) ),
		array( "typcn typcn-brush" => __( "Brush", "js_composer" ) ),
		array( "typcn typcn-business-card" => __( "Business Card", "js_composer" ) ),
		array( "typcn typcn-calculator" => __( "Calculator", "js_composer" ) ),
		array( "typcn typcn-calendar-outline" => __( "Calendar Outline", "js_composer" ) ),
		array( "typcn typcn-calendar" => __( "Calendar", "js_composer" ) ),
		array( "typcn typcn-camera-outline" => __( "Camera Outline", "js_composer" ) ),
		array( "typcn typcn-camera" => __( "Camera", "js_composer" ) ),
		array( "typcn typcn-cancel-outline" => __( "Cancel Outline", "js_composer" ) ),
		array( "typcn typcn-cancel" => __( "Cancel", "js_composer" ) ),
		array( "typcn typcn-chart-area-outline" => __( "Chart Area Outline", "js_composer" ) ),
		array( "typcn typcn-chart-area" => __( "Chart Area", "js_composer" ) ),
		array( "typcn typcn-chart-bar-outline" => __( "Chart Bar Outline", "js_composer" ) ),
		array( "typcn typcn-chart-bar" => __( "Chart Bar", "js_composer" ) ),
		array( "typcn typcn-chart-line-outline" => __( "Chart Line Outline", "js_composer" ) ),
		array( "typcn typcn-chart-line" => __( "Chart Line", "js_composer" ) ),
		array( "typcn typcn-chart-pie-outline" => __( "Chart Pie Outline", "js_composer" ) ),
		array( "typcn typcn-chart-pie" => __( "Chart Pie", "js_composer" ) ),
		array( "typcn typcn-chevron-left-outline" => __( "Chevron Left Outline", "js_composer" ) ),
		array( "typcn typcn-chevron-left" => __( "Chevron Left", "js_composer" ) ),
		array( "typcn typcn-chevron-right-outline" => __( "Chevron Right Outline", "js_composer" ) ),
		array( "typcn typcn-chevron-right" => __( "Chevron Right", "js_composer" ) ),
		array( "typcn typcn-clipboard" => __( "Clipboard", "js_composer" ) ),
		array( "typcn typcn-cloud-storage" => __( "Cloud Storage", "js_composer" ) ),
		array( "typcn typcn-cloud-storage-outline" => __( "Cloud Storage Outline", "js_composer" ) ),
		array( "typcn typcn-code-outline" => __( "Code Outline", "js_composer" ) ),
		array( "typcn typcn-code" => __( "Code", "js_composer" ) ),
		array( "typcn typcn-coffee" => __( "Coffee", "js_composer" ) ),
		array( "typcn typcn-cog-outline" => __( "Cog Outline", "js_composer" ) ),
		array( "typcn typcn-cog" => __( "Cog", "js_composer" ) ),
		array( "typcn typcn-compass" => __( "Compass", "js_composer" ) ),
		array( "typcn typcn-contacts" => __( "Contacts", "js_composer" ) ),
		array( "typcn typcn-credit-card" => __( "Credit Card", "js_composer" ) ),
		array( "typcn typcn-css3" => __( "Css3", "js_composer" ) ),
		array( "typcn typcn-database" => __( "Database", "js_composer" ) ),
		array( "typcn typcn-delete-outline" => __( "Delete Outline", "js_composer" ) ),
		array( "typcn typcn-delete" => __( "Delete", "js_composer" ) ),
		array( "typcn typcn-device-desktop" => __( "Device Desktop", "js_composer" ) ),
		array( "typcn typcn-device-laptop" => __( "Device Laptop", "js_composer" ) ),
		array( "typcn typcn-device-phone" => __( "Device Phone", "js_composer" ) ),
		array( "typcn typcn-device-tablet" => __( "Device Tablet", "js_composer" ) ),
		array( "typcn typcn-directions" => __( "Directions", "js_composer" ) ),
		array( "typcn typcn-divide-outline" => __( "Divide Outline", "js_composer" ) ),
		array( "typcn typcn-divide" => __( "Divide", "js_composer" ) ),
		array( "typcn typcn-document-add" => __( "Document Add", "js_composer" ) ),
		array( "typcn typcn-document-delete" => __( "Document Delete", "js_composer" ) ),
		array( "typcn typcn-document-text" => __( "Document Text", "js_composer" ) ),
		array( "typcn typcn-document" => __( "Document", "js_composer" ) ),
		array( "typcn typcn-download-outline" => __( "Download Outline", "js_composer" ) ),
		array( "typcn typcn-download" => __( "Download", "js_composer" ) ),
		array( "typcn typcn-dropbox" => __( "Dropbox", "js_composer" ) ),
		array( "typcn typcn-edit" => __( "Edit", "js_composer" ) ),
		array( "typcn typcn-eject-outline" => __( "Eject Outline", "js_composer" ) ),
		array( "typcn typcn-eject" => __( "Eject", "js_composer" ) ),
		array( "typcn typcn-equals-outline" => __( "Equals Outline", "js_composer" ) ),
		array( "typcn typcn-equals" => __( "Equals", "js_composer" ) ),
		array( "typcn typcn-export-outline" => __( "Export Outline", "js_composer" ) ),
		array( "typcn typcn-export" => __( "Export", "js_composer" ) ),
		array( "typcn typcn-eye-outline" => __( "Eye Outline", "js_composer" ) ),
		array( "typcn typcn-eye" => __( "Eye", "js_composer" ) ),
		array( "typcn typcn-feather" => __( "Feather", "js_composer" ) ),
		array( "typcn typcn-film" => __( "Film", "js_composer" ) ),
		array( "typcn typcn-filter" => __( "Filter", "js_composer" ) ),
		array( "typcn typcn-flag-outline" => __( "Flag Outline", "js_composer" ) ),
		array( "typcn typcn-flag" => __( "Flag", "js_composer" ) ),
		array( "typcn typcn-flash-outline" => __( "Flash Outline", "js_composer" ) ),
		array( "typcn typcn-flash" => __( "Flash", "js_composer" ) ),
		array( "typcn typcn-flow-children" => __( "Flow Children", "js_composer" ) ),
		array( "typcn typcn-flow-merge" => __( "Flow Merge", "js_composer" ) ),
		array( "typcn typcn-flow-parallel" => __( "Flow Parallel", "js_composer" ) ),
		array( "typcn typcn-flow-switch" => __( "Flow Switch", "js_composer" ) ),
		array( "typcn typcn-folder-add" => __( "Folder Add", "js_composer" ) ),
		array( "typcn typcn-folder-delete" => __( "Folder Delete", "js_composer" ) ),
		array( "typcn typcn-folder-open" => __( "Folder Open", "js_composer" ) ),
		array( "typcn typcn-folder" => __( "Folder", "js_composer" ) ),
		array( "typcn typcn-gift" => __( "Gift", "js_composer" ) ),
		array( "typcn typcn-globe-outline" => __( "Globe Outline", "js_composer" ) ),
		array( "typcn typcn-globe" => __( "Globe", "js_composer" ) ),
		array( "typcn typcn-group-outline" => __( "Group Outline", "js_composer" ) ),
		array( "typcn typcn-group" => __( "Group", "js_composer" ) ),
		array( "typcn typcn-headphones" => __( "Headphones", "js_composer" ) ),
		array( "typcn typcn-heart-full-outline" => __( "Heart Full Outline", "js_composer" ) ),
		array( "typcn typcn-heart-half-outline" => __( "Heart Half Outline", "js_composer" ) ),
		array( "typcn typcn-heart-outline" => __( "Heart Outline", "js_composer" ) ),
		array( "typcn typcn-heart" => __( "Heart", "js_composer" ) ),
		array( "typcn typcn-home-outline" => __( "Home Outline", "js_composer" ) ),
		array( "typcn typcn-home" => __( "Home", "js_composer" ) ),
		array( "typcn typcn-html5" => __( "Html5", "js_composer" ) ),
		array( "typcn typcn-image-outline" => __( "Image Outline", "js_composer" ) ),
		array( "typcn typcn-image" => __( "Image", "js_composer" ) ),
		array( "typcn typcn-infinity-outline" => __( "Infinity Outline", "js_composer" ) ),
		array( "typcn typcn-infinity" => __( "Infinity", "js_composer" ) ),
		array( "typcn typcn-info-large-outline" => __( "Info Large Outline", "js_composer" ) ),
		array( "typcn typcn-info-large" => __( "Info Large", "js_composer" ) ),
		array( "typcn typcn-info-outline" => __( "Info Outline", "js_composer" ) ),
		array( "typcn typcn-info" => __( "Info", "js_composer" ) ),
		array( "typcn typcn-input-checked-outline" => __( "Input Checked Outline", "js_composer" ) ),
		array( "typcn typcn-input-checked" => __( "Input Checked", "js_composer" ) ),
		array( "typcn typcn-key-outline" => __( "Key Outline", "js_composer" ) ),
		array( "typcn typcn-key" => __( "Key", "js_composer" ) ),
		array( "typcn typcn-keyboard" => __( "Keyboard", "js_composer" ) ),
		array( "typcn typcn-leaf" => __( "Leaf", "js_composer" ) ),
		array( "typcn typcn-lightbulb" => __( "Lightbulb", "js_composer" ) ),
		array( "typcn typcn-link-outline" => __( "Link Outline", "js_composer" ) ),
		array( "typcn typcn-link" => __( "Link", "js_composer" ) ),
		array( "typcn typcn-location-arrow-outline" => __( "Location Arrow Outline", "js_composer" ) ),
		array( "typcn typcn-location-arrow" => __( "Location Arrow", "js_composer" ) ),
		array( "typcn typcn-location-outline" => __( "Location Outline", "js_composer" ) ),
		array( "typcn typcn-location" => __( "Location", "js_composer" ) ),
		array( "typcn typcn-lock-closed-outline" => __( "Lock Closed Outline", "js_composer" ) ),
		array( "typcn typcn-lock-closed" => __( "Lock Closed", "js_composer" ) ),
		array( "typcn typcn-lock-open-outline" => __( "Lock Open Outline", "js_composer" ) ),
		array( "typcn typcn-lock-open" => __( "Lock Open", "js_composer" ) ),
		array( "typcn typcn-mail" => __( "Mail", "js_composer" ) ),
		array( "typcn typcn-map" => __( "Map", "js_composer" ) ),
		array( "typcn typcn-media-eject-outline" => __( "Media Eject Outline", "js_composer" ) ),
		array( "typcn typcn-media-eject" => __( "Media Eject", "js_composer" ) ),
		array( "typcn typcn-media-fast-forward-outline" => __( "Media Fast Forward Outline", "js_composer" ) ),
		array( "typcn typcn-media-fast-forward" => __( "Media Fast Forward", "js_composer" ) ),
		array( "typcn typcn-media-pause-outline" => __( "Media Pause Outline", "js_composer" ) ),
		array( "typcn typcn-media-pause" => __( "Media Pause", "js_composer" ) ),
		array( "typcn typcn-media-play-outline" => __( "Media Play Outline", "js_composer" ) ),
		array( "typcn typcn-media-play-reverse-outline" => __( "Media Play Reverse Outline", "js_composer" ) ),
		array( "typcn typcn-media-play-reverse" => __( "Media Play Reverse", "js_composer" ) ),
		array( "typcn typcn-media-play" => __( "Media Play", "js_composer" ) ),
		array( "typcn typcn-media-record-outline" => __( "Media Record Outline", "js_composer" ) ),
		array( "typcn typcn-media-record" => __( "Media Record", "js_composer" ) ),
		array( "typcn typcn-media-rewind-outline" => __( "Media Rewind Outline", "js_composer" ) ),
		array( "typcn typcn-media-rewind" => __( "Media Rewind", "js_composer" ) ),
		array( "typcn typcn-media-stop-outline" => __( "Media Stop Outline", "js_composer" ) ),
		array( "typcn typcn-media-stop" => __( "Media Stop", "js_composer" ) ),
		array( "typcn typcn-message-typing" => __( "Message Typing", "js_composer" ) ),
		array( "typcn typcn-message" => __( "Message", "js_composer" ) ),
		array( "typcn typcn-messages" => __( "Messages", "js_composer" ) ),
		array( "typcn typcn-microphone-outline" => __( "Microphone Outline", "js_composer" ) ),
		array( "typcn typcn-microphone" => __( "Microphone", "js_composer" ) ),
		array( "typcn typcn-minus-outline" => __( "Minus Outline", "js_composer" ) ),
		array( "typcn typcn-minus" => __( "Minus", "js_composer" ) ),
		array( "typcn typcn-mortar-board" => __( "Mortar Board", "js_composer" ) ),
		array( "typcn typcn-news" => __( "News", "js_composer" ) ),
		array( "typcn typcn-notes-outline" => __( "Notes Outline", "js_composer" ) ),
		array( "typcn typcn-notes" => __( "Notes", "js_composer" ) ),
		array( "typcn typcn-pen" => __( "Pen", "js_composer" ) ),
		array( "typcn typcn-pencil" => __( "Pencil", "js_composer" ) ),
		array( "typcn typcn-phone-outline" => __( "Phone Outline", "js_composer" ) ),
		array( "typcn typcn-phone" => __( "Phone", "js_composer" ) ),
		array( "typcn typcn-pi-outline" => __( "Pi Outline", "js_composer" ) ),
		array( "typcn typcn-pi" => __( "Pi", "js_composer" ) ),
		array( "typcn typcn-pin-outline" => __( "Pin Outline", "js_composer" ) ),
		array( "typcn typcn-pin" => __( "Pin", "js_composer" ) ),
		array( "typcn typcn-pipette" => __( "Pipette", "js_composer" ) ),
		array( "typcn typcn-plane-outline" => __( "Plane Outline", "js_composer" ) ),
		array( "typcn typcn-plane" => __( "Plane", "js_composer" ) ),
		array( "typcn typcn-plug" => __( "Plug", "js_composer" ) ),
		array( "typcn typcn-plus-outline" => __( "Plus Outline", "js_composer" ) ),
		array( "typcn typcn-plus" => __( "Plus", "js_composer" ) ),
		array( "typcn typcn-point-of-interest-outline" => __( "Point Of Interest Outline", "js_composer" ) ),
		array( "typcn typcn-point-of-interest" => __( "Point Of Interest", "js_composer" ) ),
		array( "typcn typcn-power-outline" => __( "Power Outline", "js_composer" ) ),
		array( "typcn typcn-power" => __( "Power", "js_composer" ) ),
		array( "typcn typcn-printer" => __( "Printer", "js_composer" ) ),
		array( "typcn typcn-puzzle-outline" => __( "Puzzle Outline", "js_composer" ) ),
		array( "typcn typcn-puzzle" => __( "Puzzle", "js_composer" ) ),
		array( "typcn typcn-radar-outline" => __( "Radar Outline", "js_composer" ) ),
		array( "typcn typcn-radar" => __( "Radar", "js_composer" ) ),
		array( "typcn typcn-refresh-outline" => __( "Refresh Outline", "js_composer" ) ),
		array( "typcn typcn-refresh" => __( "Refresh", "js_composer" ) ),
		array( "typcn typcn-rss-outline" => __( "Rss Outline", "js_composer" ) ),
		array( "typcn typcn-rss" => __( "Rss", "js_composer" ) ),
		array( "typcn typcn-scissors-outline" => __( "Scissors Outline", "js_composer" ) ),
		array( "typcn typcn-scissors" => __( "Scissors", "js_composer" ) ),
		array( "typcn typcn-shopping-bag" => __( "Shopping Bag", "js_composer" ) ),
		array( "typcn typcn-shopping-cart" => __( "Shopping Cart", "js_composer" ) ),
		array( "typcn typcn-social-at-circular" => __( "Social At Circular", "js_composer" ) ),
		array( "typcn typcn-social-dribbble-circular" => __( "Social Dribbble Circular", "js_composer" ) ),
		array( "typcn typcn-social-dribbble" => __( "Social Dribbble", "js_composer" ) ),
		array( "typcn typcn-social-facebook-circular" => __( "Social Facebook Circular", "js_composer" ) ),
		array( "typcn typcn-social-facebook" => __( "Social Facebook", "js_composer" ) ),
		array( "typcn typcn-social-flickr-circular" => __( "Social Flickr Circular", "js_composer" ) ),
		array( "typcn typcn-social-flickr" => __( "Social Flickr", "js_composer" ) ),
		array( "typcn typcn-social-github-circular" => __( "Social Github Circular", "js_composer" ) ),
		array( "typcn typcn-social-github" => __( "Social Github", "js_composer" ) ),
		array( "typcn typcn-social-google-plus-circular" => __( "Social Google Plus Circular", "js_composer" ) ),
		array( "typcn typcn-social-google-plus" => __( "Social Google Plus", "js_composer" ) ),
		array( "typcn typcn-social-instagram-circular" => __( "Social Instagram Circular", "js_composer" ) ),
		array( "typcn typcn-social-instagram" => __( "Social Instagram", "js_composer" ) ),
		array( "typcn typcn-social-last-fm-circular" => __( "Social Last Fm Circular", "js_composer" ) ),
		array( "typcn typcn-social-last-fm" => __( "Social Last Fm", "js_composer" ) ),
		array( "typcn typcn-social-linkedin-circular" => __( "Social Linkedin Circular", "js_composer" ) ),
		array( "typcn typcn-social-linkedin" => __( "Social Linkedin", "js_composer" ) ),
		array( "typcn typcn-social-pinterest-circular" => __( "Social Pinterest Circular", "js_composer" ) ),
		array( "typcn typcn-social-pinterest" => __( "Social Pinterest", "js_composer" ) ),
		array( "typcn typcn-social-skype-outline" => __( "Social Skype Outline", "js_composer" ) ),
		array( "typcn typcn-social-skype" => __( "Social Skype", "js_composer" ) ),
		array( "typcn typcn-social-tumbler-circular" => __( "Social Tumbler Circular", "js_composer" ) ),
		array( "typcn typcn-social-tumbler" => __( "Social Tumbler", "js_composer" ) ),
		array( "typcn typcn-social-twitter-circular" => __( "Social Twitter Circular", "js_composer" ) ),
		array( "typcn typcn-social-twitter" => __( "Social Twitter", "js_composer" ) ),
		array( "typcn typcn-social-vimeo-circular" => __( "Social Vimeo Circular", "js_composer" ) ),
		array( "typcn typcn-social-vimeo" => __( "Social Vimeo", "js_composer" ) ),
		array( "typcn typcn-social-youtube-circular" => __( "Social Youtube Circular", "js_composer" ) ),
		array( "typcn typcn-social-youtube" => __( "Social Youtube", "js_composer" ) ),
		array( "typcn typcn-sort-alphabetically-outline" => __( "Sort Alphabetically Outline", "js_composer" ) ),
		array( "typcn typcn-sort-alphabetically" => __( "Sort Alphabetically", "js_composer" ) ),
		array( "typcn typcn-sort-numerically-outline" => __( "Sort Numerically Outline", "js_composer" ) ),
		array( "typcn typcn-sort-numerically" => __( "Sort Numerically", "js_composer" ) ),
		array( "typcn typcn-spanner-outline" => __( "Spanner Outline", "js_composer" ) ),
		array( "typcn typcn-spanner" => __( "Spanner", "js_composer" ) ),
		array( "typcn typcn-spiral" => __( "Spiral", "js_composer" ) ),
		array( "typcn typcn-star-full-outline" => __( "Star Full Outline", "js_composer" ) ),
		array( "typcn typcn-star-half-outline" => __( "Star Half Outline", "js_composer" ) ),
		array( "typcn typcn-star-half" => __( "Star Half", "js_composer" ) ),
		array( "typcn typcn-star-outline" => __( "Star Outline", "js_composer" ) ),
		array( "typcn typcn-star" => __( "Star", "js_composer" ) ),
		array( "typcn typcn-starburst-outline" => __( "Starburst Outline", "js_composer" ) ),
		array( "typcn typcn-starburst" => __( "Starburst", "js_composer" ) ),
		array( "typcn typcn-stopwatch" => __( "Stopwatch", "js_composer" ) ),
		array( "typcn typcn-support" => __( "Support", "js_composer" ) ),
		array( "typcn typcn-tabs-outline" => __( "Tabs Outline", "js_composer" ) ),
		array( "typcn typcn-tag" => __( "Tag", "js_composer" ) ),
		array( "typcn typcn-tags" => __( "Tags", "js_composer" ) ),
		array( "typcn typcn-th-large-outline" => __( "Th Large Outline", "js_composer" ) ),
		array( "typcn typcn-th-large" => __( "Th Large", "js_composer" ) ),
		array( "typcn typcn-th-list-outline" => __( "Th List Outline", "js_composer" ) ),
		array( "typcn typcn-th-list" => __( "Th List", "js_composer" ) ),
		array( "typcn typcn-th-menu-outline" => __( "Th Menu Outline", "js_composer" ) ),
		array( "typcn typcn-th-menu" => __( "Th Menu", "js_composer" ) ),
		array( "typcn typcn-th-small-outline" => __( "Th Small Outline", "js_composer" ) ),
		array( "typcn typcn-th-small" => __( "Th Small", "js_composer" ) ),
		array( "typcn typcn-thermometer" => __( "Thermometer", "js_composer" ) ),
		array( "typcn typcn-thumbs-down" => __( "Thumbs Down", "js_composer" ) ),
		array( "typcn typcn-thumbs-ok" => __( "Thumbs Ok", "js_composer" ) ),
		array( "typcn typcn-thumbs-up" => __( "Thumbs Up", "js_composer" ) ),
		array( "typcn typcn-tick-outline" => __( "Tick Outline", "js_composer" ) ),
		array( "typcn typcn-tick" => __( "Tick", "js_composer" ) ),
		array( "typcn typcn-ticket" => __( "Ticket", "js_composer" ) ),
		array( "typcn typcn-time" => __( "Time", "js_composer" ) ),
		array( "typcn typcn-times-outline" => __( "Times Outline", "js_composer" ) ),
		array( "typcn typcn-times" => __( "Times", "js_composer" ) ),
		array( "typcn typcn-trash" => __( "Trash", "js_composer" ) ),
		array( "typcn typcn-tree" => __( "Tree", "js_composer" ) ),
		array( "typcn typcn-upload-outline" => __( "Upload Outline", "js_composer" ) ),
		array( "typcn typcn-upload" => __( "Upload", "js_composer" ) ),
		array( "typcn typcn-user-add-outline" => __( "User Add Outline", "js_composer" ) ),
		array( "typcn typcn-user-add" => __( "User Add", "js_composer" ) ),
		array( "typcn typcn-user-delete-outline" => __( "User Delete Outline", "js_composer" ) ),
		array( "typcn typcn-user-delete" => __( "User Delete", "js_composer" ) ),
		array( "typcn typcn-user-outline" => __( "User Outline", "js_composer" ) ),
		array( "typcn typcn-user" => __( "User", "js_composer" ) ),
		array( "typcn typcn-vendor-android" => __( "Vendor Android", "js_composer" ) ),
		array( "typcn typcn-vendor-apple" => __( "Vendor Apple", "js_composer" ) ),
		array( "typcn typcn-vendor-microsoft" => __( "Vendor Microsoft", "js_composer" ) ),
		array( "typcn typcn-video-outline" => __( "Video Outline", "js_composer" ) ),
		array( "typcn typcn-video" => __( "Video", "js_composer" ) ),
		array( "typcn typcn-volume-down" => __( "Volume Down", "js_composer" ) ),
		array( "typcn typcn-volume-mute" => __( "Volume Mute", "js_composer" ) ),
		array( "typcn typcn-volume-up" => __( "Volume Up", "js_composer" ) ),
		array( "typcn typcn-volume" => __( "Volume", "js_composer" ) ),
		array( "typcn typcn-warning-outline" => __( "Warning Outline", "js_composer" ) ),
		array( "typcn typcn-warning" => __( "Warning", "js_composer" ) ),
		array( "typcn typcn-watch" => __( "Watch", "js_composer" ) ),
		array( "typcn typcn-waves-outline" => __( "Waves Outline", "js_composer" ) ),
		array( "typcn typcn-waves" => __( "Waves", "js_composer" ) ),
		array( "typcn typcn-weather-cloudy" => __( "Weather Cloudy", "js_composer" ) ),
		array( "typcn typcn-weather-downpour" => __( "Weather Downpour", "js_composer" ) ),
		array( "typcn typcn-weather-night" => __( "Weather Night", "js_composer" ) ),
		array( "typcn typcn-weather-partly-sunny" => __( "Weather Partly Sunny", "js_composer" ) ),
		array( "typcn typcn-weather-shower" => __( "Weather Shower", "js_composer" ) ),
		array( "typcn typcn-weather-snow" => __( "Weather Snow", "js_composer" ) ),
		array( "typcn typcn-weather-stormy" => __( "Weather Stormy", "js_composer" ) ),
		array( "typcn typcn-weather-sunny" => __( "Weather Sunny", "js_composer" ) ),
		array( "typcn typcn-weather-windy-cloudy" => __( "Weather Windy Cloudy", "js_composer" ) ),
		array( "typcn typcn-weather-windy" => __( "Weather Windy", "js_composer" ) ),
		array( "typcn typcn-wi-fi-outline" => __( "Wi Fi Outline", "js_composer" ) ),
		array( "typcn typcn-wi-fi" => __( "Wi Fi", "js_composer" ) ),
		array( "typcn typcn-wine" => __( "Wine", "js_composer" ) ),
		array( "typcn typcn-world-outline" => __( "World Outline", "js_composer" ) ),
		array( "typcn typcn-world" => __( "World", "js_composer" ) ),
		array( "typcn typcn-zoom-in-outline" => __( "Zoom In Outline", "js_composer" ) ),
		array( "typcn typcn-zoom-in" => __( "Zoom In", "js_composer" ) ),
		array( "typcn typcn-zoom-out-outline" => __( "Zoom Out Outline", "js_composer" ) ),
		array( "typcn typcn-zoom-out" => __( "Zoom Out", "js_composer" ) ),
		array( "typcn typcn-zoom-outline" => __( "Zoom Outline", "js_composer" ) ),
		array( "typcn typcn-zoom" => __( "Zoom", "js_composer" ) ),
	);

	return array_merge( $icons, $typicons_icons );
}

add_filter( 'vc_iconpicker-type-entypo', 'vc_iconpicker_type_entypo' );
/**
 * Entypo icons from github.com/danielbruce/entypo
 *
 * @param $icons - taken from filter - vc_map param field settings['source'] provided icons (default empty array).
 * If array categorized it will auto-enable category dropdown
 *
 * @since 4.4
 * @return array - of icons for iconpicker, can be categorized, or not.
 */
function vc_iconpicker_type_entypo( $icons ) {
	$entypo_icons = array(
		array( "entypo-icon entypo-icon-note" => __( "Note", "js_composer" ) ),
		array( "entypo-icon entypo-icon-note-beamed" => __( "Note Beamed", "js_composer" ) ),
		array( "entypo-icon entypo-icon-music" => __( "Music", "js_composer" ) ),
		array( "entypo-icon entypo-icon-search" => __( "Search", "js_composer" ) ),
		array( "entypo-icon entypo-icon-flashlight" => __( "Flashlight", "js_composer" ) ),
		array( "entypo-icon entypo-icon-mail" => __( "Mail", "js_composer" ) ),
		array( "entypo-icon entypo-icon-heart" => __( "Heart", "js_composer" ) ),
		array( "entypo-icon entypo-icon-heart-empty" => __( "Heart Empty", "js_composer" ) ),
		array( "entypo-icon entypo-icon-star" => __( "Star", "js_composer" ) ),
		array( "entypo-icon entypo-icon-star-empty" => __( "Star Empty", "js_composer" ) ),
		array( "entypo-icon entypo-icon-user" => __( "User", "js_composer" ) ),
		array( "entypo-icon entypo-icon-users" => __( "Users", "js_composer" ) ),
		array( "entypo-icon entypo-icon-user-add" => __( "User Add", "js_composer" ) ),
		array( "entypo-icon entypo-icon-video" => __( "Video", "js_composer" ) ),
		array( "entypo-icon entypo-icon-picture" => __( "Picture", "js_composer" ) ),
		array( "entypo-icon entypo-icon-camera" => __( "Camera", "js_composer" ) ),
		array( "entypo-icon entypo-icon-layout" => __( "Layout", "js_composer" ) ),
		array( "entypo-icon entypo-icon-menu" => __( "Menu", "js_composer" ) ),
		array( "entypo-icon entypo-icon-check" => __( "Check", "js_composer" ) ),
		array( "entypo-icon entypo-icon-cancel" => __( "Cancel", "js_composer" ) ),
		array( "entypo-icon entypo-icon-cancel-circled" => __( "Cancel Circled", "js_composer" ) ),
		array( "entypo-icon entypo-icon-cancel-squared" => __( "Cancel Squared", "js_composer" ) ),
		array( "entypo-icon entypo-icon-plus" => __( "Plus", "js_composer" ) ),
		array( "entypo-icon entypo-icon-plus-circled" => __( "Plus Circled", "js_composer" ) ),
		array( "entypo-icon entypo-icon-plus-squared" => __( "Plus Squared", "js_composer" ) ),
		array( "entypo-icon entypo-icon-minus" => __( "Minus", "js_composer" ) ),
		array( "entypo-icon entypo-icon-minus-circled" => __( "Minus Circled", "js_composer" ) ),
		array( "entypo-icon entypo-icon-minus-squared" => __( "Minus Squared", "js_composer" ) ),
		array( "entypo-icon entypo-icon-help" => __( "Help", "js_composer" ) ),
		array( "entypo-icon entypo-icon-help-circled" => __( "Help Circled", "js_composer" ) ),
		array( "entypo-icon entypo-icon-info" => __( "Info", "js_composer" ) ),
		array( "entypo-icon entypo-icon-info-circled" => __( "Info Circled", "js_composer" ) ),
		array( "entypo-icon entypo-icon-back" => __( "Back", "js_composer" ) ),
		array( "entypo-icon entypo-icon-home" => __( "Home", "js_composer" ) ),
		array( "entypo-icon entypo-icon-link" => __( "Link", "js_composer" ) ),
		array( "entypo-icon entypo-icon-attach" => __( "Attach", "js_composer" ) ),
		array( "entypo-icon entypo-icon-lock" => __( "Lock", "js_composer" ) ),
		array( "entypo-icon entypo-icon-lock-open" => __( "Lock Open", "js_composer" ) ),
		array( "entypo-icon entypo-icon-eye" => __( "Eye", "js_composer" ) ),
		array( "entypo-icon entypo-icon-tag" => __( "Tag", "js_composer" ) ),
		array( "entypo-icon entypo-icon-bookmark" => __( "Bookmark", "js_composer" ) ),
		array( "entypo-icon entypo-icon-bookmarks" => __( "Bookmarks", "js_composer" ) ),
		array( "entypo-icon entypo-icon-flag" => __( "Flag", "js_composer" ) ),
		array( "entypo-icon entypo-icon-thumbs-up" => __( "Thumbs Up", "js_composer" ) ),
		array( "entypo-icon entypo-icon-thumbs-down" => __( "Thumbs Down", "js_composer" ) ),
		array( "entypo-icon entypo-icon-download" => __( "Download", "js_composer" ) ),
		array( "entypo-icon entypo-icon-upload" => __( "Upload", "js_composer" ) ),
		array( "entypo-icon entypo-icon-upload-cloud" => __( "Upload Cloud", "js_composer" ) ),
		array( "entypo-icon entypo-icon-reply" => __( "Reply", "js_composer" ) ),
		array( "entypo-icon entypo-icon-reply-all" => __( "Reply All", "js_composer" ) ),
		array( "entypo-icon entypo-icon-forward" => __( "Forward", "js_composer" ) ),
		array( "entypo-icon entypo-icon-quote" => __( "Quote", "js_composer" ) ),
		array( "entypo-icon entypo-icon-code" => __( "Code", "js_composer" ) ),
		array( "entypo-icon entypo-icon-export" => __( "Export", "js_composer" ) ),
		array( "entypo-icon entypo-icon-pencil" => __( "Pencil", "js_composer" ) ),
		array( "entypo-icon entypo-icon-feather" => __( "Feather", "js_composer" ) ),
		array( "entypo-icon entypo-icon-print" => __( "Print", "js_composer" ) ),
		array( "entypo-icon entypo-icon-retweet" => __( "Retweet", "js_composer" ) ),
		array( "entypo-icon entypo-icon-keyboard" => __( "Keyboard", "js_composer" ) ),
		array( "entypo-icon entypo-icon-comment" => __( "Comment", "js_composer" ) ),
		array( "entypo-icon entypo-icon-chat" => __( "Chat", "js_composer" ) ),
		array( "entypo-icon entypo-icon-bell" => __( "Bell", "js_composer" ) ),
		array( "entypo-icon entypo-icon-attention" => __( "Attention", "js_composer" ) ),
		array( "entypo-icon entypo-icon-alert" => __( "Alert", "js_composer" ) ),
		array( "entypo-icon entypo-icon-vcard" => __( "Vcard", "js_composer" ) ),
		array( "entypo-icon entypo-icon-address" => __( "Address", "js_composer" ) ),
		array( "entypo-icon entypo-icon-location" => __( "Location", "js_composer" ) ),
		array( "entypo-icon entypo-icon-map" => __( "Map", "js_composer" ) ),
		array( "entypo-icon entypo-icon-direction" => __( "Direction", "js_composer" ) ),
		array( "entypo-icon entypo-icon-compass" => __( "Compass", "js_composer" ) ),
		array( "entypo-icon entypo-icon-cup" => __( "Cup", "js_composer" ) ),
		array( "entypo-icon entypo-icon-trash" => __( "Trash", "js_composer" ) ),
		array( "entypo-icon entypo-icon-doc" => __( "Doc", "js_composer" ) ),
		array( "entypo-icon entypo-icon-docs" => __( "Docs", "js_composer" ) ),
		array( "entypo-icon entypo-icon-doc-landscape" => __( "Doc Landscape", "js_composer" ) ),
		array( "entypo-icon entypo-icon-doc-text" => __( "Doc Text", "js_composer" ) ),
		array( "entypo-icon entypo-icon-doc-text-inv" => __( "Doc Text Inv", "js_composer" ) ),
		array( "entypo-icon entypo-icon-newspaper" => __( "Newspaper", "js_composer" ) ),
		array( "entypo-icon entypo-icon-book-open" => __( "Book Open", "js_composer" ) ),
		array( "entypo-icon entypo-icon-book" => __( "Book", "js_composer" ) ),
		array( "entypo-icon entypo-icon-folder" => __( "Folder", "js_composer" ) ),
		array( "entypo-icon entypo-icon-archive" => __( "Archive", "js_composer" ) ),
		array( "entypo-icon entypo-icon-box" => __( "Box", "js_composer" ) ),
		array( "entypo-icon entypo-icon-rss" => __( "Rss", "js_composer" ) ),
		array( "entypo-icon entypo-icon-phone" => __( "Phone", "js_composer" ) ),
		array( "entypo-icon entypo-icon-cog" => __( "Cog", "js_composer" ) ),
		array( "entypo-icon entypo-icon-tools" => __( "Tools", "js_composer" ) ),
		array( "entypo-icon entypo-icon-share" => __( "Share", "js_composer" ) ),
		array( "entypo-icon entypo-icon-shareable" => __( "Shareable", "js_composer" ) ),
		array( "entypo-icon entypo-icon-basket" => __( "Basket", "js_composer" ) ),
		array( "entypo-icon entypo-icon-bag" => __( "Bag", "js_composer" ) ),
		array( "entypo-icon entypo-icon-calendar" => __( "Calendar", "js_composer" ) ),
		array( "entypo-icon entypo-icon-login" => __( "Login", "js_composer" ) ),
		array( "entypo-icon entypo-icon-logout" => __( "Logout", "js_composer" ) ),
		array( "entypo-icon entypo-icon-mic" => __( "Mic", "js_composer" ) ),
		array( "entypo-icon entypo-icon-mute" => __( "Mute", "js_composer" ) ),
		array( "entypo-icon entypo-icon-sound" => __( "Sound", "js_composer" ) ),
		array( "entypo-icon entypo-icon-volume" => __( "Volume", "js_composer" ) ),
		array( "entypo-icon entypo-icon-clock" => __( "Clock", "js_composer" ) ),
		array( "entypo-icon entypo-icon-hourglass" => __( "Hourglass", "js_composer" ) ),
		array( "entypo-icon entypo-icon-lamp" => __( "Lamp", "js_composer" ) ),
		array( "entypo-icon entypo-icon-light-down" => __( "Light Down", "js_composer" ) ),
		array( "entypo-icon entypo-icon-light-up" => __( "Light Up", "js_composer" ) ),
		array( "entypo-icon entypo-icon-adjust" => __( "Adjust", "js_composer" ) ),
		array( "entypo-icon entypo-icon-block" => __( "Block", "js_composer" ) ),
		array( "entypo-icon entypo-icon-resize-full" => __( "Resize Full", "js_composer" ) ),
		array( "entypo-icon entypo-icon-resize-small" => __( "Resize Small", "js_composer" ) ),
		array( "entypo-icon entypo-icon-popup" => __( "Popup", "js_composer" ) ),
		array( "entypo-icon entypo-icon-publish" => __( "Publish", "js_composer" ) ),
		array( "entypo-icon entypo-icon-window" => __( "Window", "js_composer" ) ),
		array( "entypo-icon entypo-icon-arrow-combo" => __( "Arrow Combo", "js_composer" ) ),
		array( "entypo-icon entypo-icon-down-circled" => __( "Down Circled", "js_composer" ) ),
		array( "entypo-icon entypo-icon-left-circled" => __( "Left Circled", "js_composer" ) ),
		array( "entypo-icon entypo-icon-right-circled" => __( "Right Circled", "js_composer" ) ),
		array( "entypo-icon entypo-icon-up-circled" => __( "Up Circled", "js_composer" ) ),
		array( "entypo-icon entypo-icon-down-open" => __( "Down Open", "js_composer" ) ),
		array( "entypo-icon entypo-icon-left-open" => __( "Left Open", "js_composer" ) ),
		array( "entypo-icon entypo-icon-right-open" => __( "Right Open", "js_composer" ) ),
		array( "entypo-icon entypo-icon-up-open" => __( "Up Open", "js_composer" ) ),
		array( "entypo-icon entypo-icon-down-open-mini" => __( "Down Open Mini", "js_composer" ) ),
		array( "entypo-icon entypo-icon-left-open-mini" => __( "Left Open Mini", "js_composer" ) ),
		array( "entypo-icon entypo-icon-right-open-mini" => __( "Right Open Mini", "js_composer" ) ),
		array( "entypo-icon entypo-icon-up-open-mini" => __( "Up Open Mini", "js_composer" ) ),
		array( "entypo-icon entypo-icon-down-open-big" => __( "Down Open Big", "js_composer" ) ),
		array( "entypo-icon entypo-icon-left-open-big" => __( "Left Open Big", "js_composer" ) ),
		array( "entypo-icon entypo-icon-right-open-big" => __( "Right Open Big", "js_composer" ) ),
		array( "entypo-icon entypo-icon-up-open-big" => __( "Up Open Big", "js_composer" ) ),
		array( "entypo-icon entypo-icon-down" => __( "Down", "js_composer" ) ),
		array( "entypo-icon entypo-icon-left" => __( "Left", "js_composer" ) ),
		array( "entypo-icon entypo-icon-right" => __( "Right", "js_composer" ) ),
		array( "entypo-icon entypo-icon-up" => __( "Up", "js_composer" ) ),
		array( "entypo-icon entypo-icon-down-dir" => __( "Down Dir", "js_composer" ) ),
		array( "entypo-icon entypo-icon-left-dir" => __( "Left Dir", "js_composer" ) ),
		array( "entypo-icon entypo-icon-right-dir" => __( "Right Dir", "js_composer" ) ),
		array( "entypo-icon entypo-icon-up-dir" => __( "Up Dir", "js_composer" ) ),
		array( "entypo-icon entypo-icon-down-bold" => __( "Down Bold", "js_composer" ) ),
		array( "entypo-icon entypo-icon-left-bold" => __( "Left Bold", "js_composer" ) ),
		array( "entypo-icon entypo-icon-right-bold" => __( "Right Bold", "js_composer" ) ),
		array( "entypo-icon entypo-icon-up-bold" => __( "Up Bold", "js_composer" ) ),
		array( "entypo-icon entypo-icon-down-thin" => __( "Down Thin", "js_composer" ) ),
		array( "entypo-icon entypo-icon-left-thin" => __( "Left Thin", "js_composer" ) ),
		array( "entypo-icon entypo-icon-right-thin" => __( "Right Thin", "js_composer" ) ),
		array( "entypo-icon entypo-icon-up-thin" => __( "Up Thin", "js_composer" ) ),
		array( "entypo-icon entypo-icon-ccw" => __( "Ccw", "js_composer" ) ),
		array( "entypo-icon entypo-icon-cw" => __( "Cw", "js_composer" ) ),
		array( "entypo-icon entypo-icon-arrows-ccw" => __( "Arrows Ccw", "js_composer" ) ),
		array( "entypo-icon entypo-icon-level-down" => __( "Level Down", "js_composer" ) ),
		array( "entypo-icon entypo-icon-level-up" => __( "Level Up", "js_composer" ) ),
		array( "entypo-icon entypo-icon-shuffle" => __( "Shuffle", "js_composer" ) ),
		array( "entypo-icon entypo-icon-loop" => __( "Loop", "js_composer" ) ),
		array( "entypo-icon entypo-icon-switch" => __( "Switch", "js_composer" ) ),
		array( "entypo-icon entypo-icon-play" => __( "Play", "js_composer" ) ),
		array( "entypo-icon entypo-icon-stop" => __( "Stop", "js_composer" ) ),
		array( "entypo-icon entypo-icon-pause" => __( "Pause", "js_composer" ) ),
		array( "entypo-icon entypo-icon-record" => __( "Record", "js_composer" ) ),
		array( "entypo-icon entypo-icon-to-end" => __( "To End", "js_composer" ) ),
		array( "entypo-icon entypo-icon-to-start" => __( "To Start", "js_composer" ) ),
		array( "entypo-icon entypo-icon-fast-forward" => __( "Fast Forward", "js_composer" ) ),
		array( "entypo-icon entypo-icon-fast-backward" => __( "Fast Backward", "js_composer" ) ),
		array( "entypo-icon entypo-icon-progress-0" => __( "Progress 0", "js_composer" ) ),
		array( "entypo-icon entypo-icon-progress-1" => __( "Progress 1", "js_composer" ) ),
		array( "entypo-icon entypo-icon-progress-2" => __( "Progress 2", "js_composer" ) ),
		array( "entypo-icon entypo-icon-progress-3" => __( "Progress 3", "js_composer" ) ),
		array( "entypo-icon entypo-icon-target" => __( "Target", "js_composer" ) ),
		array( "entypo-icon entypo-icon-palette" => __( "Palette", "js_composer" ) ),
		array( "entypo-icon entypo-icon-list" => __( "List", "js_composer" ) ),
		array( "entypo-icon entypo-icon-list-add" => __( "List Add", "js_composer" ) ),
		array( "entypo-icon entypo-icon-signal" => __( "Signal", "js_composer" ) ),
		array( "entypo-icon entypo-icon-trophy" => __( "Trophy", "js_composer" ) ),
		array( "entypo-icon entypo-icon-battery" => __( "Battery", "js_composer" ) ),
		array( "entypo-icon entypo-icon-back-in-time" => __( "Back In Time", "js_composer" ) ),
		array( "entypo-icon entypo-icon-monitor" => __( "Monitor", "js_composer" ) ),
		array( "entypo-icon entypo-icon-mobile" => __( "Mobile", "js_composer" ) ),
		array( "entypo-icon entypo-icon-network" => __( "Network", "js_composer" ) ),
		array( "entypo-icon entypo-icon-cd" => __( "Cd", "js_composer" ) ),
		array( "entypo-icon entypo-icon-inbox" => __( "Inbox", "js_composer" ) ),
		array( "entypo-icon entypo-icon-install" => __( "Install", "js_composer" ) ),
		array( "entypo-icon entypo-icon-globe" => __( "Globe", "js_composer" ) ),
		array( "entypo-icon entypo-icon-cloud" => __( "Cloud", "js_composer" ) ),
		array( "entypo-icon entypo-icon-cloud-thunder" => __( "Cloud Thunder", "js_composer" ) ),
		array( "entypo-icon entypo-icon-flash" => __( "Flash", "js_composer" ) ),
		array( "entypo-icon entypo-icon-moon" => __( "Moon", "js_composer" ) ),
		array( "entypo-icon entypo-icon-flight" => __( "Flight", "js_composer" ) ),
		array( "entypo-icon entypo-icon-paper-plane" => __( "Paper Plane", "js_composer" ) ),
		array( "entypo-icon entypo-icon-leaf" => __( "Leaf", "js_composer" ) ),
		array( "entypo-icon entypo-icon-lifebuoy" => __( "Lifebuoy", "js_composer" ) ),
		array( "entypo-icon entypo-icon-mouse" => __( "Mouse", "js_composer" ) ),
		array( "entypo-icon entypo-icon-briefcase" => __( "Briefcase", "js_composer" ) ),
		array( "entypo-icon entypo-icon-suitcase" => __( "Suitcase", "js_composer" ) ),
		array( "entypo-icon entypo-icon-dot" => __( "Dot", "js_composer" ) ),
		array( "entypo-icon entypo-icon-dot-2" => __( "Dot 2", "js_composer" ) ),
		array( "entypo-icon entypo-icon-dot-3" => __( "Dot 3", "js_composer" ) ),
		array( "entypo-icon entypo-icon-brush" => __( "Brush", "js_composer" ) ),
		array( "entypo-icon entypo-icon-magnet" => __( "Magnet", "js_composer" ) ),
		array( "entypo-icon entypo-icon-infinity" => __( "Infinity", "js_composer" ) ),
		array( "entypo-icon entypo-icon-erase" => __( "Erase", "js_composer" ) ),
		array( "entypo-icon entypo-icon-chart-pie" => __( "Chart Pie", "js_composer" ) ),
		array( "entypo-icon entypo-icon-chart-line" => __( "Chart Line", "js_composer" ) ),
		array( "entypo-icon entypo-icon-chart-bar" => __( "Chart Bar", "js_composer" ) ),
		array( "entypo-icon entypo-icon-chart-area" => __( "Chart Area", "js_composer" ) ),
		array( "entypo-icon entypo-icon-tape" => __( "Tape", "js_composer" ) ),
		array( "entypo-icon entypo-icon-graduation-cap" => __( "Graduation Cap", "js_composer" ) ),
		array( "entypo-icon entypo-icon-language" => __( "Language", "js_composer" ) ),
		array( "entypo-icon entypo-icon-ticket" => __( "Ticket", "js_composer" ) ),
		array( "entypo-icon entypo-icon-water" => __( "Water", "js_composer" ) ),
		array( "entypo-icon entypo-icon-droplet" => __( "Droplet", "js_composer" ) ),
		array( "entypo-icon entypo-icon-air" => __( "Air", "js_composer" ) ),
		array( "entypo-icon entypo-icon-credit-card" => __( "Credit Card", "js_composer" ) ),
		array( "entypo-icon entypo-icon-floppy" => __( "Floppy", "js_composer" ) ),
		array( "entypo-icon entypo-icon-clipboard" => __( "Clipboard", "js_composer" ) ),
		array( "entypo-icon entypo-icon-megaphone" => __( "Megaphone", "js_composer" ) ),
		array( "entypo-icon entypo-icon-database" => __( "Database", "js_composer" ) ),
		array( "entypo-icon entypo-icon-drive" => __( "Drive", "js_composer" ) ),
		array( "entypo-icon entypo-icon-bucket" => __( "Bucket", "js_composer" ) ),
		array( "entypo-icon entypo-icon-thermometer" => __( "Thermometer", "js_composer" ) ),
		array( "entypo-icon entypo-icon-key" => __( "Key", "js_composer" ) ),
		array( "entypo-icon entypo-icon-flow-cascade" => __( "Flow Cascade", "js_composer" ) ),
		array( "entypo-icon entypo-icon-flow-branch" => __( "Flow Branch", "js_composer" ) ),
		array( "entypo-icon entypo-icon-flow-tree" => __( "Flow Tree", "js_composer" ) ),
		array( "entypo-icon entypo-icon-flow-line" => __( "Flow Line", "js_composer" ) ),
		array( "entypo-icon entypo-icon-flow-parallel" => __( "Flow Parallel", "js_composer" ) ),
		array( "entypo-icon entypo-icon-rocket" => __( "Rocket", "js_composer" ) ),
		array( "entypo-icon entypo-icon-gauge" => __( "Gauge", "js_composer" ) ),
		array( "entypo-icon entypo-icon-traffic-cone" => __( "Traffic Cone", "js_composer" ) ),
		array( "entypo-icon entypo-icon-cc" => __( "Cc", "js_composer" ) ),
		array( "entypo-icon entypo-icon-cc-by" => __( "Cc By", "js_composer" ) ),
		array( "entypo-icon entypo-icon-cc-nc" => __( "Cc Nc", "js_composer" ) ),
		array( "entypo-icon entypo-icon-cc-nc-eu" => __( "Cc Nc Eu", "js_composer" ) ),
		array( "entypo-icon entypo-icon-cc-nc-jp" => __( "Cc Nc Jp", "js_composer" ) ),
		array( "entypo-icon entypo-icon-cc-sa" => __( "Cc Sa", "js_composer" ) ),
		array( "entypo-icon entypo-icon-cc-nd" => __( "Cc Nd", "js_composer" ) ),
		array( "entypo-icon entypo-icon-cc-pd" => __( "Cc Pd", "js_composer" ) ),
		array( "entypo-icon entypo-icon-cc-zero" => __( "Cc Zero", "js_composer" ) ),
		array( "entypo-icon entypo-icon-cc-share" => __( "Cc Share", "js_composer" ) ),
		array( "entypo-icon entypo-icon-cc-remix" => __( "Cc Remix", "js_composer" ) ),
		array( "entypo-icon entypo-icon-github" => __( "Github", "js_composer" ) ),
		array( "entypo-icon entypo-icon-github-circled" => __( "Github Circled", "js_composer" ) ),
		array( "entypo-icon entypo-icon-flickr" => __( "Flickr", "js_composer" ) ),
		array( "entypo-icon entypo-icon-flickr-circled" => __( "Flickr Circled", "js_composer" ) ),
		array( "entypo-icon entypo-icon-vimeo" => __( "Vimeo", "js_composer" ) ),
		array( "entypo-icon entypo-icon-vimeo-circled" => __( "Vimeo Circled", "js_composer" ) ),
		array( "entypo-icon entypo-icon-twitter" => __( "Twitter", "js_composer" ) ),
		array( "entypo-icon entypo-icon-twitter-circled" => __( "Twitter Circled", "js_composer" ) ),
		array( "entypo-icon entypo-icon-facebook" => __( "Facebook", "js_composer" ) ),
		array( "entypo-icon entypo-icon-facebook-circled" => __( "Facebook Circled", "js_composer" ) ),
		array( "entypo-icon entypo-icon-facebook-squared" => __( "Facebook Squared", "js_composer" ) ),
		array( "entypo-icon entypo-icon-gplus" => __( "Gplus", "js_composer" ) ),
		array( "entypo-icon entypo-icon-gplus-circled" => __( "Gplus Circled", "js_composer" ) ),
		array( "entypo-icon entypo-icon-pinterest" => __( "Pinterest", "js_composer" ) ),
		array( "entypo-icon entypo-icon-pinterest-circled" => __( "Pinterest Circled", "js_composer" ) ),
		array( "entypo-icon entypo-icon-tumblr" => __( "Tumblr", "js_composer" ) ),
		array( "entypo-icon entypo-icon-tumblr-circled" => __( "Tumblr Circled", "js_composer" ) ),
		array( "entypo-icon entypo-icon-linkedin" => __( "Linkedin", "js_composer" ) ),
		array( "entypo-icon entypo-icon-linkedin-circled" => __( "Linkedin Circled", "js_composer" ) ),
		array( "entypo-icon entypo-icon-dribbble" => __( "Dribbble", "js_composer" ) ),
		array( "entypo-icon entypo-icon-dribbble-circled" => __( "Dribbble Circled", "js_composer" ) ),
		array( "entypo-icon entypo-icon-stumbleupon" => __( "Stumbleupon", "js_composer" ) ),
		array( "entypo-icon entypo-icon-stumbleupon-circled" => __( "Stumbleupon Circled", "js_composer" ) ),
		array( "entypo-icon entypo-icon-lastfm" => __( "Lastfm", "js_composer" ) ),
		array( "entypo-icon entypo-icon-lastfm-circled" => __( "Lastfm Circled", "js_composer" ) ),
		array( "entypo-icon entypo-icon-rdio" => __( "Rdio", "js_composer" ) ),
		array( "entypo-icon entypo-icon-rdio-circled" => __( "Rdio Circled", "js_composer" ) ),
		array( "entypo-icon entypo-icon-spotify" => __( "Spotify", "js_composer" ) ),
		array( "entypo-icon entypo-icon-spotify-circled" => __( "Spotify Circled", "js_composer" ) ),
		array( "entypo-icon entypo-icon-qq" => __( "Qq", "js_composer" ) ),
		array( "entypo-icon entypo-icon-instagrem" => __( "Instagrem", "js_composer" ) ),
		array( "entypo-icon entypo-icon-dropbox" => __( "Dropbox", "js_composer" ) ),
		array( "entypo-icon entypo-icon-evernote" => __( "Evernote", "js_composer" ) ),
		array( "entypo-icon entypo-icon-flattr" => __( "Flattr", "js_composer" ) ),
		array( "entypo-icon entypo-icon-skype" => __( "Skype", "js_composer" ) ),
		array( "entypo-icon entypo-icon-skype-circled" => __( "Skype Circled", "js_composer" ) ),
		array( "entypo-icon entypo-icon-renren" => __( "Renren", "js_composer" ) ),
		array( "entypo-icon entypo-icon-sina-weibo" => __( "Sina Weibo", "js_composer" ) ),
		array( "entypo-icon entypo-icon-paypal" => __( "Paypal", "js_composer" ) ),
		array( "entypo-icon entypo-icon-picasa" => __( "Picasa", "js_composer" ) ),
		array( "entypo-icon entypo-icon-soundcloud" => __( "Soundcloud", "js_composer" ) ),
		array( "entypo-icon entypo-icon-mixi" => __( "Mixi", "js_composer" ) ),
		array( "entypo-icon entypo-icon-behance" => __( "Behance", "js_composer" ) ),
		array( "entypo-icon entypo-icon-google-circles" => __( "Google Circles", "js_composer" ) ),
		array( "entypo-icon entypo-icon-vkontakte" => __( "Vkontakte", "js_composer" ) ),
		array( "entypo-icon entypo-icon-smashing" => __( "Smashing", "js_composer" ) ),
		array( "entypo-icon entypo-icon-sweden" => __( "Sweden", "js_composer" ) ),
		array( "entypo-icon entypo-icon-db-shape" => __( "Db Shape", "js_composer" ) ),
		array( "entypo-icon entypo-icon-logo-db" => __( "Logo Db", "js_composer" ) ),
	);

	return array_merge( $icons, $entypo_icons );
}

add_filter( 'vc_iconpicker-type-linecons', 'vc_iconpicker_type_linecons' );

/**
 * Linecons icons from fontello.com
 *
 * @param $icons - taken from filter - vc_map param field settings['source'] provided icons (default empty array).
 * If array categorized it will auto-enable category dropdown
 *
 * @since 4.4
 * @return array - of icons for iconpicker, can be categorized, or not.
 */
function vc_iconpicker_type_linecons( $icons ) {
	$linecons_icons = array(
		array( "vc_li vc_li-heart" => __( "Heart", "js_composer" ) ),
		array( "vc_li vc_li-cloud" => __( "Cloud", "js_composer" ) ),
		array( "vc_li vc_li-star" => __( "Star", "js_composer" ) ),
		array( "vc_li vc_li-tv" => __( "Tv", "js_composer" ) ),
		array( "vc_li vc_li-sound" => __( "Sound", "js_composer" ) ),
		array( "vc_li vc_li-video" => __( "Video", "js_composer" ) ),
		array( "vc_li vc_li-trash" => __( "Trash", "js_composer" ) ),
		array( "vc_li vc_li-user" => __( "User", "js_composer" ) ),
		array( "vc_li vc_li-key" => __( "Key", "js_composer" ) ),
		array( "vc_li vc_li-search" => __( "Search", "js_composer" ) ),
		array( "vc_li vc_li-settings" => __( "Settings", "js_composer" ) ),
		array( "vc_li vc_li-camera" => __( "Camera", "js_composer" ) ),
		array( "vc_li vc_li-tag" => __( "Tag", "js_composer" ) ),
		array( "vc_li vc_li-lock" => __( "Lock", "js_composer" ) ),
		array( "vc_li vc_li-bulb" => __( "Bulb", "js_composer" ) ),
		array( "vc_li vc_li-pen" => __( "Pen", "js_composer" ) ),
		array( "vc_li vc_li-diamond" => __( "Diamond", "js_composer" ) ),
		array( "vc_li vc_li-display" => __( "Display", "js_composer" ) ),
		array( "vc_li vc_li-location" => __( "Location", "js_composer" ) ),
		array( "vc_li vc_li-eye" => __( "Eye", "js_composer" ) ),
		array( "vc_li vc_li-bubble" => __( "Bubble", "js_composer" ) ),
		array( "vc_li vc_li-stack" => __( "Stack", "js_composer" ) ),
		array( "vc_li vc_li-cup" => __( "Cup", "js_composer" ) ),
		array( "vc_li vc_li-phone" => __( "Phone", "js_composer" ) ),
		array( "vc_li vc_li-news" => __( "News", "js_composer" ) ),
		array( "vc_li vc_li-mail" => __( "Mail", "js_composer" ) ),
		array( "vc_li vc_li-like" => __( "Like", "js_composer" ) ),
		array( "vc_li vc_li-photo" => __( "Photo", "js_composer" ) ),
		array( "vc_li vc_li-note" => __( "Note", "js_composer" ) ),
		array( "vc_li vc_li-clock" => __( "Clock", "js_composer" ) ),
		array( "vc_li vc_li-paperplane" => __( "Paperplane", "js_composer" ) ),
		array( "vc_li vc_li-params" => __( "Params", "js_composer" ) ),
		array( "vc_li vc_li-banknote" => __( "Banknote", "js_composer" ) ),
		array( "vc_li vc_li-data" => __( "Data", "js_composer" ) ),
		array( "vc_li vc_li-music" => __( "Music", "js_composer" ) ),
		array( "vc_li vc_li-megaphone" => __( "Megaphone", "js_composer" ) ),
		array( "vc_li vc_li-study" => __( "Study", "js_composer" ) ),
		array( "vc_li vc_li-lab" => __( "Lab", "js_composer" ) ),
		array( "vc_li vc_li-food" => __( "Food", "js_composer" ) ),
		array( "vc_li vc_li-t-shirt" => __( "T Shirt", "js_composer" ) ),
		array( "vc_li vc_li-fire" => __( "Fire", "js_composer" ) ),
		array( "vc_li vc_li-clip" => __( "Clip", "js_composer" ) ),
		array( "vc_li vc_li-shop" => __( "Shop", "js_composer" ) ),
		array( "vc_li vc_li-calendar" => __( "Calendar", "js_composer" ) ),
		array( "vc_li vc_li-vallet" => __( "Vallet", "js_composer" ) ),
		array( "vc_li vc_li-vynil" => __( "Vynil", "js_composer" ) ),
		array( "vc_li vc_li-truck" => __( "Truck", "js_composer" ) ),
		array( "vc_li vc_li-world" => __( "World", "js_composer" ) ),
	);

	return array_merge( $icons, $linecons_icons );
}